package com.bxm.newidea.recommend.handler.forum;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.config.NewsCacheThresholdConfig;
import com.bxm.newidea.constant.AppConstant;
import com.bxm.newidea.constant.RedisKeyConstant;
import com.bxm.newidea.domain.ForumPostMapper;
import com.bxm.newidea.enums.OperationLocationEnum;
import com.bxm.newidea.param.ForumParam;
import com.bxm.newidea.param.ForumQueryParam;
import com.bxm.newidea.properties.RecommendProperties;
import com.bxm.newidea.recommend.framework.AbstractForumRecommender;
import com.bxm.newidea.util.ObjectUtil;
import com.bxm.newidea.vo.ForumRecommend;
import com.bxm.newidea.vo.ForumTopPost;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 论坛板块推荐器：
 * 按时间降序排序推荐帖子
 * </p>
 *
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2019/4/4
 */
@Service
public class ForumCommonRecommender extends AbstractForumRecommender {

    private static final Logger logger = LoggerFactory.getLogger(ForumCommonRecommender.class);

    @Autowired
    private RedisSetAdapter redisSetAdapter;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private ForumPostMapper forumPostMapper;

    @Autowired
    private RedisListAdapter redisListAdapter;

    @Autowired
    private RecommendProperties recommendProperties;

    public ForumCommonRecommender() {
        super(1, 1);
    }

    @Override
    public List<Long> syncRecommend(ForumParam forumParam) {
        LinkedList<Long> result = Lists.newLinkedList();

        String name = OperationLocationEnum.getName(forumParam.getOperationId());

        // 已推荐redis key
        KeyGenerator readRecordKey = RedisKeyConstant.FORUM_RECOMMENDED.copy().appendKey(forumParam.getUserId()).appendKey(forumParam.getPlatform()).appendKey(name).appendKey("already");

        // 最后一次推荐记录
        KeyGenerator lastRecommendedKey = RedisKeyConstant.FORUM_RECOMMENDED.copy().appendKey(forumParam.getUserId()).appendKey(forumParam.getPlatform()).appendKey(name).appendKey("last");

        // 已推荐记录
        Set<Long> readRecordList = Sets.newHashSet();
        // 上一次推荐的最后一条记录时间
        String lastRecommendedTime = "";
        LinkedList<Long> recommendList = new LinkedList<>();
        if (null == forumParam.getActionType() || AppConstant.ACTION_TYPE_DOWN == forumParam.getActionType()) {
            redisStringAdapter.remove(readRecordKey);
            redisStringAdapter.remove(lastRecommendedKey);
            recommendList = this.getRecommendPost(forumParam, readRecordList, readRecordKey,forumParam.getPlatform());
        } else {
            readRecordList.addAll(redisSetAdapter.getAllMembers(readRecordKey, Long.class));
            lastRecommendedTime = redisStringAdapter.getString(lastRecommendedKey);
        }

        int leftNum = forumParam.getPageSize() - recommendList.size();
        if (leftNum > 0) {
            //拼装查询参数
            ForumQueryParam param = getQueryParam(forumParam, lastRecommendedTime);

            //数据库查询可以推荐数据
            List<ForumRecommend> forumPostList = forumPostMapper.findForumPostList(param);

            //过滤去重
            recommendedFilter(forumPostList, result, readRecordKey, lastRecommendedKey, readRecordList, leftNum);
        }

        //穿插推荐帖子
        this.interspersedRecommendPost(recommendList, result);

        return result;
    }

    /**
     * 穿插推荐帖子
     *
     * @param recommendPostIds
     * @param postIds
     */
    private void interspersedRecommendPost(LinkedList<Long> recommendPostIds, List<Long> postIds) {
        if (!org.springframework.util.CollectionUtils.isEmpty(recommendPostIds)) {
            int count = recommendPostIds.size();
            int index = 0;
            for (int i = 0; i < count; i++) {
                if (index >= postIds.size()) {
                    recommendPostIds.addLast(recommendPostIds.get(i));
                } else {
                    postIds.add(index, recommendPostIds.get(i));
                    index = index + 2;
                }
            }
        }
    }

    /**
     * 获取每日推荐
     *
     * @param forumParam     请求参数
     * @param readRecordKey  结果集
     * @param readRecordList 已阅读记录
     */
    private LinkedList<Long> getRecommendPost(ForumParam forumParam, Set<Long> readRecordList, KeyGenerator readRecordKey,int platform) {
        LinkedList<Long> result = new LinkedList<>();
        if (forumParam.getOperationId() == OperationLocationEnum.FORUM_HOME_PAGE.getCode()) {
            Set<Long> recommendRecordList = Sets.newHashSet(readRecordList);
            //本地圈是否已经加载过一次推荐帖子
            KeyGenerator dailyRecommendKey = RedisKeyConstant.FORUM_RECOMMENDED.copy()
                    .appendKey(forumParam.getUserId())
                    .appendKey(platform)
                    .appendKey(OperationLocationEnum.getName(forumParam.getOperationId()))
                    .appendKey(DateFormatUtils.format(new Date(), "yyyy-MM-dd"));
            if (!redisStringAdapter.hasKey(dailyRecommendKey)) {
                // 首页已推荐记录
                KeyGenerator homeReadRecordKey = RedisKeyConstant.FORUM_RECOMMENDED.copy()
                        .appendKey(forumParam.getUserId())
                        .appendKey(platform)
                        .appendKey(OperationLocationEnum.HOME_PAGE.getName())
                        .appendKey("already");
                recommendRecordList.addAll(redisSetAdapter.getAllMembers(homeReadRecordKey, Long.class));

                List<ForumRecommend> forumPostIdList = forumPostMapper.findRecommendForumPost(forumParam.getAreaCode(), 100);
                recommendedFilter(forumPostIdList, result, homeReadRecordKey, readRecordKey, recommendRecordList, readRecordList);
                redisStringAdapter.set(dailyRecommendKey, 1, DateUtils.getCurSeconds());
            }
        }
        return result;
    }

    /**
     * 拼装参数
     *
     * @param forumParam
     * @param lastRecommendedTime 最后阅读时间
     * @return
     */
    private ForumQueryParam getQueryParam(ForumParam forumParam, String lastRecommendedTime) {
        Integer isBrilliant = OperationLocationEnum.FORUM_PLATE_ELITE.getCode() == forumParam.getOperationId() ? 1 : null;
        ForumQueryParam param = new ForumQueryParam();
        param.setAreaCode(forumParam.getAreaCode());
        param.setBrilliant(isBrilliant);
        param.setUserId(forumParam.getUserId());
        param.setForumId(forumParam.getForumId());
        param.setLastTime(lastRecommendedTime);
        param.setTopicId(forumParam.getTopicId());
        param.setSize(forumParam.getPageSize() + 50);
        if (forumParam.getOperationId() == OperationLocationEnum.FORUM_HOME_PAGE.getCode() && CollectionUtils.isNotEmpty(recommendProperties.getFilterForumId())) {
            param.setFilterForumId(recommendProperties.getFilterForumId());
        }
        return param;
    }

    /**
     * 过滤已阅读的
     *
     * @param originList        原始集
     * @param resultList        结果集
     * @param homeReadRecordKey 阅读时间记录key（用来存记录）
     * @param readRecordKey     最后阅读时间key（用来存记录）
     * @param readRecordList    阅读记录
     */
    private void recommendedFilter(List<ForumRecommend> originList,
                                   List<Long> resultList,
                                   KeyGenerator homeReadRecordKey,
                                   KeyGenerator readRecordKey,
                                   Set<Long> recommendReadRecordList,
                                   Set<Long> readRecordList) {
        //用来记录最后一个数据的时间
        for (ForumRecommend forumRecommend : originList) {

            //1.初始判断是否与之前的最新推荐池id或者本次的固定推荐池id重复，如果不重复则将其加入结果，并加入临时推荐记录做去重准备
            if (!recommendReadRecordList.contains(forumRecommend.getId())) {
                recommendReadRecordList.add(forumRecommend.getId());
                resultList.add(forumRecommend.getId());
                readRecordList.add(forumRecommend.getId());
            }
            //2.一旦数量达标，则退出循环
            if (resultList.size() >= 3) {
                break;
            }
        }
        logger.debug("[recommendedFilter]已取得{}条优质帖子推荐数据，功成圆满", resultList.size());
        if (CollectionUtils.isNotEmpty(resultList)) {
            //3.存记录
            redisSetAdapter.add(homeReadRecordKey, resultList.toArray());
            redisSetAdapter.expire(homeReadRecordKey, NewsCacheThresholdConfig.MIX_EXPIRE_RECOMMENDED_RECORD);

            redisSetAdapter.add(readRecordKey, resultList.toArray());
            redisSetAdapter.expire(readRecordKey, NewsCacheThresholdConfig.MIX_EXPIRE_RECOMMENDED_RECORD);
        }
    }


    /**
     * 过滤已阅读的
     *
     * @param originList         原始集
     * @param resultList         结果集
     * @param recommendRecordKey 阅读时间记录key（用来存记录）
     * @param lastRecommendedKey 最后阅读时间key（用来存记录）
     * @param readRecordList     阅读记录
     * @param num                本次获取结果集数量
     */
    private void recommendedFilter(List<ForumRecommend> originList,
                                   LinkedList<Long> resultList,
                                   KeyGenerator recommendRecordKey,
                                   KeyGenerator lastRecommendedKey,
                                   Set<Long> readRecordList,
                                   int num) {
        //用来记录最后一个数据的时间
        Date lastReadTime = null;
        for (ForumRecommend forumRecommend : originList) {

            //1.初始判断是否与之前的最新推荐池id或者本次的固定推荐池id重复，如果不重复则将其加入结果，并加入临时推荐记录做去重准备
            if (!readRecordList.contains(forumRecommend.getId())) {
                resultList.add(forumRecommend.getId());
                readRecordList.add(forumRecommend.getId());
                lastReadTime = forumRecommend.getDisplayTime();
            }
            //2.一旦数量达标，则退出循环
            if (resultList.size() >= num) {
                break;
            }
        }
        logger.debug("[recommendedFilter]已取得{}条帖子推荐数据，功成圆满", resultList.size());
        if (CollectionUtils.isNotEmpty(resultList)) {
            //3.存记录
            redisSetAdapter.add(recommendRecordKey, resultList.toArray());
            redisSetAdapter.expire(recommendRecordKey, NewsCacheThresholdConfig.MIX_EXPIRE_RECOMMENDED_RECORD);
            //4.请求三条推荐数据的时候不需要存最后阅读时间
            if (Objects.nonNull(lastReadTime)) {
                redisStringAdapter.set(lastRecommendedKey, DateUtils.formatDateTime(lastReadTime));
                redisStringAdapter.expire(lastRecommendedKey, NewsCacheThresholdConfig.MIX_EXPIRE_RECOMMENDED_RECORD);
            }
        }
    }


}
