package com.bxm.newidea.recommend.handler.forum;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.constant.AppConstant;
import com.bxm.newidea.constant.RedisKeyConstant;
import com.bxm.newidea.domain.ForumPostMapper;
import com.bxm.newidea.enums.OperationLocationEnum;
import com.bxm.newidea.param.ForumParam;
import com.bxm.newidea.param.ForumQueryParam;
import com.bxm.newidea.recommend.framework.AbstractForumRecommender;
import com.bxm.newidea.util.ObjectUtil;
import com.bxm.newidea.vo.ForumRecommend;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <p>
 *     最热话题帖子推荐器：
 *         话题下，按帖子热度降序推荐
 * </p>
 *
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2019/4/4
 */
@Service
public class ForumTopicHotRecommender extends AbstractForumRecommender {
    private static final Logger LOG = LoggerFactory.getLogger(ForumTopicHotRecommender.class);
    @Autowired
    private RedisSetAdapter    redisSetAdapter;
    @Autowired
    private RedisStringAdapter redisStringAdapter;
    @Autowired
    private ForumPostMapper    forumPostMapper;

    public ForumTopicHotRecommender() {
        super(1, 1);
    }

    @Override
    public List<Long> syncRecommend(ForumParam forumParam) {
        Long userId = forumParam.getUserId();
        Integer operationId = forumParam.getOperationId();
        // 精华板块已推荐redis key
        KeyGenerator recommendedKey = RedisKeyConstant.FORUM_RECOMMENDED.copy().appendKey(forumParam.getUserId()).appendKey(forumParam.getPlatform()).appendKey(OperationLocationEnum.getName(operationId)).appendKey("already");
        // 最后一次推荐记录
        KeyGenerator lastRecommendedKey = RedisKeyConstant.FORUM_RECOMMENDED.copy().appendKey(forumParam.getUserId()).appendKey(forumParam.getPlatform()).appendKey(OperationLocationEnum.getName(operationId)).appendKey("last");
        // 已推荐记录
        Set<Long> recommendedIds = null;
        // 上一次推荐的最后一条记录时间
        Integer lastRecommendedTime = 0;
        if(null == forumParam.getActionType() || AppConstant.ACTION_TYPE_DOWN == forumParam.getActionType()){
            // app 行为判断,下拉更新:清空redis缓存
            redisStringAdapter.remove(recommendedKey);
            redisStringAdapter.remove(lastRecommendedKey);
        } else {
            // 获取 redis 缓存
            recommendedIds = redisSetAdapter.getAllMembers(recommendedKey,Long.class);
            lastRecommendedTime = redisStringAdapter.getInt(lastRecommendedKey);
        }
        if(null == recommendedIds){
            recommendedIds = Sets.newHashSet();
        }

        ForumQueryParam param = getQueryParam(forumParam,lastRecommendedTime,null,null);
        // 数据库查询可以推荐数据
        List<ForumRecommend> forumPostList = forumPostMapper.findForumPostList(param);
        List<Long> result = Lists.newArrayList();
        lastRecommendedTime = findMoreForumPost(result,forumPostList,recommendedIds,param,0);
        if(null != lastRecommendedTime){
            // 记录最后一次时间
            redisStringAdapter.set(lastRecommendedKey,lastRecommendedTime);
        }
        // 记录已推荐id
        if(result.size() != 0){
            Long[] idsArray = new Long[result.size()];
            redisSetAdapter.add(recommendedKey,result.toArray(idsArray));
        }

        return result;
    }

    /**
     * 获取足够数量的帖子
     * @date: 2019/4/4
     * @param result 推荐结果
     * @param forumPostList 上一次查询的结果
     * @param param
     * @return java.lang.String 最后一条的时间
     * @throws
     */
    private Integer findMoreForumPost(List<Long> result, List<ForumRecommend> forumPostList,Set<Long> recommendedIds,ForumQueryParam param, int count) {
        if(null == forumPostList || forumPostList.size() == 0 || 5 == count){
            LOG.warn("[forum topic hot]post is none");
            return null;
        }
        int postSize = forumPostList.size();
        Integer lastRecommendedTime = param.getHotScore();
        // 防止最后一个时间循环查询
        boolean hasMore = true;
        for(int j=0; j<postSize; j++){
            Long id = ObjectUtil.displayToLong(forumPostList.get(j).getId());
            // 不包含已经推荐过的id
            if(null != id && !recommendedIds.contains(id)){
                if(!result.contains(id)){
                    result.add(id);
                    recommendedIds.add(id);
                }else {
                    if(j == postSize-1) {
                        hasMore = false;
                    }
                }
                // 获取指定大小的结果
                if(result.size() >= param.getSize()){
                    return forumPostList.get(j).getHotScore();
                }
            }
            // 记录最后一个的时间
            if(j == postSize-1 && !hasMore) {
                lastRecommendedTime = null;
            } else {
                lastRecommendedTime = forumPostList.get(j).getHotScore();
            }
        }
        if(result.size() < param.getSize()){
            if(null == lastRecommendedTime){
                return param.getHotScore();
            }
            param.setHotScore(lastRecommendedTime);
            forumPostList = forumPostMapper.findForumPostList(param);
            return findMoreForumPost(result,forumPostList,recommendedIds,param,++count);
        }
        return lastRecommendedTime;
    }

    private ForumQueryParam getQueryParam(ForumParam forumParam, Integer lastRecommendedTime, Integer isBrilliant, Integer deadline) {
        ForumQueryParam param = new ForumQueryParam();
        param.setAreaCode(forumParam.getAreaCode());
        param.setBrilliant(isBrilliant);
        param.setSize(forumParam.getPageSize());
        param.setTopicId(forumParam.getTopicId());
        param.setHot(1);
        param.setHotScore(lastRecommendedTime);

        return  param;
    }
}
