package com.bxm.newidea.recommend.handler.news;

import com.alibaba.fastjson.JSON;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.config.NewsCacheThresholdConfig;
import com.bxm.newidea.constant.LocalNewsKindIdConstant;
import com.bxm.newidea.constant.RedisKeyConstant;
import com.bxm.newidea.domain.NewsMapper;
import com.bxm.newidea.integration.NewsSyncIntegrationService;
import com.bxm.newidea.param.NewsQueryParam;
import com.bxm.newidea.recommend.framework.AbstractNewsRecommender;
import com.bxm.newidea.service.WeightService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * 老版本mysql推荐策略
 * redis缓存推荐器
 * 频道推荐池
 */
@Component
public class OldMysqlNewsRecommender extends AbstractNewsRecommender {

    private NewsMapper newsMapper;

    private WeightService weightService;

    private DistributedLock distributedLock;

    @Autowired(required = false)
    public OldMysqlNewsRecommender(RedisListAdapter redisListAdapter,
                                   NewsMapper newsMapper,WeightService weightService,
                                   NewsSyncIntegrationService newsSyncIntegrationService,
                                   DistributedLock distributedLock
                                  ) {
        super(1, 4);
        this.redisListAdapter = redisListAdapter;
        this.newsMapper = newsMapper;
        this.weightService = weightService;
        this.newsSyncIntegrationService = newsSyncIntegrationService;
        this.distributedLock = distributedLock;
    }


    @Override
    protected List<Long> syncRecommend(Long userId, Integer kindId, Integer num, String areaCode) {
        long begin = System.currentTimeMillis();

        //1.根据频道id和用户id获得用户推荐缓冲池
        KeyGenerator userCacheKey = RedisKeyConstant.NEWS_NEW_RECOMMEND.copy();
        if (kindId != null) {
            userCacheKey.setKey(String.valueOf(kindId));
        }
        userCacheKey.appendKey(userId);

        //2.判断缓冲池是否有足够的数据
        long size = redisListAdapter.size(userCacheKey);
        if (size == 0) {
            return new ArrayList<>();
        }

        //3.如果缓冲池数据足够则进行数据过滤
        List<Long> resultList = new ArrayList<>();
        recommendedFilter(userCacheKey, num, resultList, userId);
        logger.debug("同步推荐耗时：{}", System.currentTimeMillis() - begin);
        return resultList;
    }


    @Override
    @Async
    protected void asyncRecommend(Long userId, Integer kindId, String areaCode) {
        logger.debug("[asyncRecommend][频道推荐]异步方法补充推荐池，用户：{}，频道：{}，地区：{}", userId, kindId, areaCode);
        long current = System.currentTimeMillis();
        NewsQueryParam newsQueryParam = new NewsQueryParam();
        //1.根据用户id，频道id，地区编号得到redis中的缓冲池
        KeyGenerator userCacheKey = getUserCacheKey(userId,kindId,areaCode, newsQueryParam);

        //2.判断缓存池中数据是否已到达阈值，如果数量不够则进行补充
        int size = redisListAdapter.size(userCacheKey).intValue();
        if (size > NewsCacheThresholdConfig.ALARM_CACHE_SIZE) {
            return;
        }

        //3.利用redis分布式锁对并发的请求进行把控，防止盲目填充缓存池数据
        String requestId = String.valueOf(nextSequence());
        if (distributedLock.lock(userCacheKey.gen(), requestId)) {

            //4.重置用户标签
            long currentRecalcu = System.currentTimeMillis();
            this.weightService.recalcu(userId);
            logger.debug("[asyncRecommend][频道推荐]给{}进行用户标签重置 花费时间{}", userId, System.currentTimeMillis() - currentRecalcu);

            //5.根据条件得到补充的数据
            List<Long> result = generateNews(newsQueryParam, userId);
            Long[] newsDtos = result.toArray(new Long[0]);

            //6.对缓冲池的数据进行重置，并保存1小时，考虑到很少用户会看1个小时以上的新闻，所以没必要存很长时间
            redisListAdapter.remove(userCacheKey);
            redisListAdapter.rightPush(userCacheKey, newsDtos);
            redisListAdapter.expire(userCacheKey, 3600);
            logger.info("[asyncRecommend][频道推荐]为{}进行频道为{}的新闻推荐 结果：{}。花费时间{}", userId, kindId == null ? "推荐" : kindId, newsDtos.length, System.currentTimeMillis() - current);

            //7.解锁
            distributedLock.unlock(userCacheKey.toString(), requestId);
        } else {
            logger.info("[asyncRecommend][频道推荐]异步向redis中填充数据仍在执行中...取消此次执行");
            return;
        }
    }

    /**
     * 得到用户缓存池key
     * @param userId
     * @param kindId
     * @param areaCode
     * @param newsQueryParam
     * @return
     */
    private KeyGenerator getUserCacheKey(Long userId, Integer kindId, String areaCode, NewsQueryParam newsQueryParam) {
        KeyGenerator userCacheKey = RedisKeyConstant.NEWS_NEW_RECOMMEND.copy();
        if (kindId != null) {
            userCacheKey.setKey(String.valueOf(kindId));
            newsQueryParam.setKindId((long) kindId);
            if (kindId.intValue() == LocalNewsKindIdConstant.LOCAL_NEWS) {
                userCacheKey.appendKey(areaCode);
                newsQueryParam.setKindId(null);
                newsQueryParam.setAreaCode(areaCode);
            }
        }
        userCacheKey.appendKey(userId);
        return userCacheKey;
    }

    /**
     * 补充满推荐缓存池
     * @param newsQueryParam
     * @param userId
     * @return
     */
    private List<Long> generateNews(NewsQueryParam newsQueryParam, Long userId) {
        //补充满推荐缓存池
        newsQueryParam.setIsHot(1);
        newsQueryParam.setUserId(userId);
        newsQueryParam.setPagesize(NewsCacheThresholdConfig.MAX_CACHE_SIZE);
        List<Long> result = newsMapper.recommendNews(newsQueryParam);
        if (result.size() < NewsCacheThresholdConfig.ALARM_CACHE_SIZE) {
            result = newsMapper.recommendNewsWithoutTime(newsQueryParam);
            logger.info("[asyncRecommend][频道推荐]由于新闻时效性导致推荐过少,所以重新进行一次推荐,参数：{}，数量：{}", JSON.toJSONString(newsQueryParam), result.size());
        }
        return result;
    }


}
