package com.bxm.newidea.recommend.handler.video;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.config.NewsCacheThresholdConfig;
import com.bxm.newidea.constant.RedisKeyConstant;
import com.bxm.newidea.domain.VideoMapper;
import com.bxm.newidea.domain.VideoRecommendMapper;
import com.bxm.newidea.dto.VideoDto;
import com.bxm.newidea.recommend.framework.AbstractVideoRecommender;
import com.google.common.hash.BloomFilter;
import com.google.common.hash.Funnels;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.ArrayList;

import java.util.List;

/**
 * 旧版本mysql小视频推荐策略
 * redis缓存推荐器
 */
@Component
public class OldMysqlVideoRecommender extends AbstractVideoRecommender {

    private Logger logger = LoggerFactory.getLogger(OldMysqlVideoRecommender.class);

    private VideoMapper videoMapper;


    @Autowired(required = false)
    public OldMysqlVideoRecommender(RedisListAdapter redisListAdapter, VideoRecommendMapper videoRecommendMapper,VideoMapper videoMapper) {
        super(1.0,0);
        this.redisListAdapter = redisListAdapter;
        this.videoRecommendMapper = videoRecommendMapper;
        this.videoMapper = videoMapper;
    }

    @Override
    protected List<VideoDto> syncRecommend(Long userId, Integer num) {
        long begin = System.currentTimeMillis();
        KeyGenerator userCacheKey = RedisKeyConstant.VIDEO_RECOMMEND.copy().appendKey(userId);
        List<VideoDto> result = redisListAdapter.leftIndex(userCacheKey, num- 1, VideoDto.class);
        if(result.isEmpty()){return result;}
        //已经推荐过的新闻key
        KeyGenerator recommendedNewsKey = RedisKeyConstant.VIDEO_RECOMMENDED.copy();
        recommendedNewsKey.appendKey(userId);
        long recommendedSize = redisListAdapter.size(recommendedNewsKey);
        List<Long> recommendedIds = redisListAdapter.leftIndex(recommendedNewsKey,recommendedSize-1,Long.class);
        //布隆过滤器去重 防止随着已经推荐过的记录膨胀 去重效率降低
        BloomFilter<Long> bloomFilter = null;
        if(recommendedIds!=null&&!recommendedIds.isEmpty()){
            bloomFilter = BloomFilter.create(Funnels.longFunnel(), recommendedIds.size(),0.0001);
            for(Long id:recommendedIds){
                bloomFilter.put(id);
            }
        }
        List<VideoDto> resultList = new ArrayList<>();
        recommend(userCacheKey,num,bloomFilter,resultList);
        logger.debug("同步推荐耗时：{}",System.currentTimeMillis()-begin);
        return resultList;
    }

    private void recommend(KeyGenerator keyGenerator,Integer num,BloomFilter<Long> bloomFilter,List<VideoDto> resultList){
        long begin = System.currentTimeMillis();
        List<VideoDto> videoList = redisListAdapter.leftIndex(keyGenerator, num - 1, VideoDto.class);
        logger.info("redis拉取列表耗时：{} 列表长：{}",System.currentTimeMillis()-begin,videoList.size());
        if(videoList.isEmpty()){
            return;
        }
        int flag = 0;
        for(VideoDto videoDto:videoList) {
            if(bloomFilter!=null&&bloomFilter.mightContain(videoDto.getId())){
                //已经推荐过
                flag++;
                continue;
            }
            resultList.add(videoDto);
        }
        long recommendSize = redisListAdapter.size(keyGenerator);
        redisListAdapter.leftTrim(keyGenerator, num, recommendSize);
        if(flag>0){
            recommend(keyGenerator,flag,bloomFilter,resultList);
        }
        logger.info("bloom去重耗时：{}",System.currentTimeMillis()-begin);
    }

    @Async
    @Override
    protected void asyncRecommend(Long userId) {
        long begin = System.currentTimeMillis();
        KeyGenerator userCacheKey = RedisKeyConstant.VIDEO_RECOMMEND.copy().appendKey(userId);
        int size =  redisListAdapter.size(userCacheKey).intValue();
        if(size> NewsCacheThresholdConfig.ALARM_CACHE_SIZE){
            //缓存池内容尚无到达触发异步推荐的阀值 则暂不进行异步推荐
            return;
        }
        //进行推荐
        List<VideoDto> result = videoMapper.getUserRecommend(userId,NewsCacheThresholdConfig.MAX_CACHE_SIZE);
        VideoDto[] videoDtos = result.toArray(new VideoDto[0]);
        //移除剩余的缓存数据，防止重复数据
        redisListAdapter.remove(userCacheKey);
        redisListAdapter.rightPush(userCacheKey, videoDtos);
        //保存30分钟
        redisListAdapter.expire(userCacheKey, 1800);
        logger.debug("给[{}]进行小视频推荐。花费时间[{}]", userId, System.currentTimeMillis() - begin);
    }
}
