package com.bxm.newidea.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.constant.AppConstant;
import com.bxm.newidea.domain.NewsMapper;
import com.bxm.newidea.dto.VideoDto;
import com.bxm.newidea.param.ForumParam;
import com.bxm.newidea.recommend.engine.NewsRecommendEngine;
import com.bxm.newidea.recommend.engine.VideoRecommendEngine;
import com.bxm.newidea.recommend.handler.forum.ForumDetailRecommender;
import com.bxm.newidea.service.RecommendService;
import com.bxm.newidea.vo.News;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Service
public class RecommendServiceImpl implements RecommendService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private NewsRecommendEngine newsRecommendEngine;

    private VideoRecommendEngine videoRecommendEngine;

    private NewsMapper newsMapper;

    @Autowired(required = false)
    public RecommendServiceImpl(NewsMapper newsMapper,
                                NewsRecommendEngine newsRecommendEngine,
                                VideoRecommendEngine videoRecommendEngine){
        this.newsMapper =newsMapper;
        this.newsRecommendEngine = newsRecommendEngine;
        this.videoRecommendEngine = videoRecommendEngine;
    }

    /**
     * 新闻推荐
     * @param userId 用户id
     * @param kindId 频道id 本地新闻时 频道id为-1
     * @param pageSize 本次获取新闻数量
     * @param areaCode 地区编码 参照国标 本地新闻时特有
     * @return 新闻列表
     */
    @Override
    public List<Long> recommend(Long userId, Integer kindId, Integer pageSize, String areaCode,Integer curPage) {
        return newsRecommendEngine.recommendNews(userId,kindId,pageSize,areaCode,curPage);
    }


    /**
     * 小视频推荐
     * @param userId 用户id
     * @param pageSize 本次获取小视频数量
     * @return 小视频列表
     */
    @Override
    public List<VideoDto> recommend( Long userId,Integer pageSize) {
        logger.info("Video recommend->userId:{},pageSize:{}",userId,pageSize);
        return videoRecommendEngine.recommendVideo(userId,pageSize);
    }

    /**
     * 新闻详情页内容推荐
     * @param userId 用户id
     * @param newsId 新闻id
     * @return 新闻id集合
     */
    @Override
    public List<Long> recommendByNewsDetail(Long userId, Long newsId,Integer size,Integer platform) {

        final Integer newsSize = AppConstant.NEWS_DETAIL_RECOMMEND_NUM;
        //如果是本地新闻则直接返回空，本地新闻不需要相关推荐
        List<Long> ids = new ArrayList<>();
        News localnews = newsMapper.selectByPrimaryKey(newsId);
        if (null != localnews) {
//            //本地新闻推荐逻辑
            if (StringUtils.isNotBlank(localnews.getAreaDetail())) {
                ids = localNewsReCommand(localnews,newsSize,userId);
            }else{
                //调用存储过程,以及制定的篇数获取详情页推荐的文章
                List<News> list = newsMapper.queryRecommendNewsList(newsId, newsSize, userId==null?0:userId);
                logger.debug("全国短期时效表查找详情推荐：{}", JSON.toJSONString(list));
                ids = list.stream().filter(Objects::nonNull).map(News::getId).collect(Collectors.toList());
            }
            //推荐平台是H5时去推荐库填充内容
            ForumDetailRecommender forumDetailRecommender = SpringContextHolder.getBean(ForumDetailRecommender.class);
            if (platform == 3){
                ids = forumDetailRecommender.getResultToH5(Sets.newHashSet(ids), localnews.getAreaDetail(),newsId);
            }else{
                ids = forumDetailRecommender.getResultToClient(Sets.newHashSet(ids),newsSize,localnews.getAreaDetail(),newsId);
            }
        }
        return ids;
    }



    private List<Long> localNewsReCommand(News localnews,Integer size, Long userId ){
        String areaDetail = localnews.getAreaDetail();
        Long province = NumberUtils.toLong(areaDetail.substring(0, 2));
        HashSet<Long> result = Sets.newHashSet();
        List<Long> shortTermList = newsMapper.queryRecommendLocalNewsList(localnews.getId(),size,userId==null ? 0 : userId,province,areaDetail);
        logger.debug("本地新闻短期时效表查找详情推荐(按标签匹配)：{}", JSON.toJSONString(shortTermList));
        result.addAll(shortTermList);
        if (result.size() < size ){
            List<Long> localNewsByTime = newsMapper.queryLocalNewShortTermByTime(province,userId,size,areaDetail,localnews.getId());
            logger.debug("审核通过的本地新闻,短期时效：{}", JSON.toJSONString(localNewsByTime));
            result.addAll(localNewsByTime);
        }
        return result.stream().limit(size).collect(Collectors.toList());
    }
}
