package com.bxm.localnews.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.config.RedisConfig;
import com.bxm.localnews.model.dto.NewsWarper;
import com.bxm.localnews.model.param.AdminNewsPageParam;
import com.bxm.localnews.model.param.NewsParam;
import com.bxm.localnews.model.vo.News;
import com.bxm.localnews.model.vo.AdminNews;
import com.bxm.localnews.model.vo.NewsDetail;
import com.bxm.localnews.model.vo.OldNews;
import com.bxm.localnews.service.NewsService;
import com.bxm.localnews.utils.ChineseToPinYinUtil;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.DisMaxQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.functionscore.FunctionScoreQueryBuilder;
import org.elasticsearch.index.query.functionscore.ScoreFunctionBuilders;
import org.elasticsearch.search.sort.ScoreSortBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.elasticsearch.core.query.*;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static org.elasticsearch.index.query.QueryBuilders.*;

//import com.bxm.localnews.domain.NewsRepository;

@Service
public class NewsServiceImpl extends BaseService implements NewsService {

    private final static Logger logger = LoggerFactory.getLogger(NewsServiceImpl.class);

//    @Autowired
//    private NewsRepository newsRepository;

    @Autowired
    private ElasticsearchOperations elasticsearchOperations;

    @Autowired
    private RedisListAdapter redisListAdapter;

    @Override
    public AdminNews selectNews(AdminNewsPageParam adminNewsPageParam) {
        Pageable pageable = PageRequest.of(adminNewsPageParam.getPageNum(), adminNewsPageParam.getPageSize());

        String key = adminNewsPageParam.getKeyword();
        Integer reviewStatus = adminNewsPageParam.getReviewStatus();
        Integer status = adminNewsPageParam.getStatus();
        Integer kindId = adminNewsPageParam.getKindId();
        Integer showLevel = adminNewsPageParam.getShowLevel();
        String areaCode = adminNewsPageParam.getAreaCode();
        String author = adminNewsPageParam.getAuthor();
        Byte isRecommend = adminNewsPageParam.getIsRecommend();

        BoolQueryBuilder queryBuilder = QueryBuilders.boolQuery();
        //如果搜索词不为空
        if (StringUtils.isNotEmpty(key)) {
            QueryBuilder keyQueryBuilder;
            keyQueryBuilder = queryPageSearch(key);
            queryBuilder.must(keyQueryBuilder);
        }

        // 是否传了id
        if (null != adminNewsPageParam.getId()) {
            queryBuilder.must(QueryBuilders.termQuery("id", adminNewsPageParam.getId()));
        }

        // 是否推荐
        if (null != isRecommend) {
            queryBuilder.must(QueryBuilders.termQuery("is_recommend", isRecommend));
        }

        // 审核状态
        if (null != reviewStatus) {
            queryBuilder.must(QueryBuilders.termQuery("review_status", reviewStatus));
        }

        // 状态
        if (null != status) {
            queryBuilder.must(QueryBuilders.termQuery("status", status));
        }

        // 频道
        if (null != kindId) {
            queryBuilder.must(QueryBuilders.termQuery("kind_id", kindId));
        }

        // 来源（如果要精确查的话要加上.keyword）
        if (null != author) {
            queryBuilder.must(QueryBuilders.termQuery("author.keyword", author));
        }

        //查询是否是本地还是全国新闻
        if (adminNewsPageParam.getDeliveryType()!=null) {
            if (adminNewsPageParam.getDeliveryType()==0) {
                queryBuilder
                        .mustNot(QueryBuilders.existsQuery("area_detail"));
            }else{
                if (StringUtils.isNotEmpty(areaCode)) {
                    queryBuilder
                            .must(QueryBuilders.termQuery("area_detail", areaCode));
                }else{
                    queryBuilder
                            .must(QueryBuilders.existsQuery("area_detail"));
                }
            }
        }

        // 判断showlevel的值对热门、置顶、爆进行筛选
        this.fillingShowLevel(queryBuilder,showLevel,areaCode);

        //查询总条数
        SearchQuery searchQueryCount = new NativeSearchQueryBuilder()
                .withQuery(queryBuilder).build();
        logger.debug("新闻搜索[查询数量]--->QueryDSL:" + searchQueryCount.getQuery().toString());

        long count = elasticsearchOperations.count(searchQueryCount, News.class);

        //进行搜索，并按时间排序
        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withQuery(queryBuilder)
                .withSort(SortBuilders.fieldSort("issue_time").order(SortOrder.DESC))
                .withSort(new ScoreSortBuilder())
                .withPageable(pageable).build();
        logger.debug("新闻搜索[分页查询]--->QueryDSL:" + searchQuery.getQuery().toString());
        List<NewsWarper> newsList = elasticsearchOperations.queryForList(searchQuery, NewsWarper.class);

        AdminNews adminNews =new AdminNews();
        adminNews.setNewsList(newsList);
        adminNews.setTotal(count);


        return adminNews;
    }

    private void fillingShowLevel(BoolQueryBuilder queryBuilder ,Integer showLevel,String areaCode){
        if (null != showLevel && 0 == showLevel) {
            queryBuilder
                    .must(QueryBuilders.termQuery("top", 2));
        } else if (null != showLevel && 1 == showLevel) {
            queryBuilder
                    .must(QueryBuilders.termQuery("hot", 2));
        }  else if (null != showLevel && 2 == showLevel) {
            queryBuilder
                    .must(QueryBuilders.termQuery("kind_top", 2));
        } else if (null != showLevel && 3 == showLevel) {
            queryBuilder
                    .must(QueryBuilders.termQuery("hot", 3));
        }
    }

    /**
     * 运营后台分页搜索
     *
     * @param key
     * @return
     */
    public QueryBuilder queryPageSearch(String key) {

        //以关键字开头(优先级最高)
        QueryBuilder ngramTitleQueryBuilder = QueryBuilders.matchQuery("title.ngram", key).analyzer("ngramSearchAnalyzer").boost(5);
        QueryBuilder q1 = QueryBuilders.boolQuery().should(ngramTitleQueryBuilder);

        //完整包含经过分析过的关键字
        QueryBuilder ikTitleQueryBuilder = QueryBuilders.matchQuery("title", key).analyzer("ikSearchAnalyzer").minimumShouldMatch("100%").boost(4);
        QueryBuilder q2 = QueryBuilders.boolQuery().should(ikTitleQueryBuilder);

        DisMaxQueryBuilder disMaxQueryBuilder = QueryBuilders.disMaxQuery();
        disMaxQueryBuilder.add(q1);
        disMaxQueryBuilder.add(q2);

        logger.debug("新闻搜索[分页搜索]");

        return disMaxQueryBuilder;
    }

    /**
     * 纯中文搜索
     *
     * @param key
     * @return
     */
    public QueryBuilder chineseSearch(String key) {

        //以关键字开头(优先级最高)
        QueryBuilder pingYinTitleQueryBuilder = QueryBuilders.matchQuery("title.ngram", key).analyzer("ngramSearchAnalyzer").boost(5);
        QueryBuilder pingYinContentQueryBuilder = QueryBuilders.matchQuery("content.ngram", key).analyzer("ngramSearchAnalyzer").boost(2);
        QueryBuilder q1 = QueryBuilders.boolQuery().should(pingYinTitleQueryBuilder).should(pingYinContentQueryBuilder);

        //完整包含经过分析过的关键字
        QueryBuilder ikTitleQueryBuilder = QueryBuilders.matchQuery("title", key).analyzer("ikSearchAnalyzer").minimumShouldMatch("100%").boost(4);
        QueryBuilder ikContentQueryBuilder = QueryBuilders.matchQuery("content", key).analyzer("ikSearchAnalyzer").minimumShouldMatch("100%").boost(2);
        QueryBuilder q2 = QueryBuilders.boolQuery().should(ikTitleQueryBuilder).should(ikContentQueryBuilder);

        DisMaxQueryBuilder disMaxQueryBuilder = QueryBuilders.disMaxQuery();
        disMaxQueryBuilder.add(q1);
        disMaxQueryBuilder.add(q2);

        logger.debug("新闻搜索[纯中文搜索]");

        return disMaxQueryBuilder;
    }

    /**
     * 中英文混合搜索
     *
     * @param key
     * @return
     */
    private QueryBuilder chineseWithEnglishOrPinyinSearchNews(String key) {
        DisMaxQueryBuilder disMaxQueryBuilder = QueryBuilders.disMaxQuery();

        //是否有中文开头，有则返回中文前缀
        String startChineseString = ChineseToPinYinUtil.getStartChineseString(key);

        /**
         * 源值搜索，不做拼音转换
         * 权重* 1.5
         */
        QueryBuilder normSearchBuilder = multiMatchQuery(key, "title.ngram", "content.ngram").analyzer("ngramSearchAnalyzer").boost(5f);

        /**
         * 拼音简写搜索
         * 1、分析key，转换为简写  case:  中国==>zg，中g==>zg，zg==>zg
         * 2、搜索匹配，必须完整匹配简写词干
         * 3、如果有中文前缀，则排序优先
         * 权重*1.0
         */
        String analysisKey = ChineseToPinYinUtil.ToFirstChar(key);
        QueryBuilder pingYinTitleQueryBuilder = QueryBuilders.termQuery("title.SPY", analysisKey).boost(1.5f);
        QueryBuilder pingYinContentQueryBuilder = QueryBuilders.termQuery("content.SPY", analysisKey).boost(1f);
        QueryBuilder pingYinSampleQueryBuilder = QueryBuilders.boolQuery().should(pingYinTitleQueryBuilder).should(pingYinContentQueryBuilder);

        /**
         * 拼音简写包含匹配，如 njdl可以查出 "城市公牛 南京东路店"，虽然非南京东路开头
         * 权重*0.8
         */
        QueryBuilder pingYinSampleContainQueryBuilder = null;
        if (analysisKey.length() > 1) {
            QueryBuilder pingYinTitleSampleContainQueryBuilder = QueryBuilders.wildcardQuery("title.SPY", "*" + analysisKey + "*").boost(1.2f);
            QueryBuilder pingYinContentSampleContainQueryBuilder = QueryBuilders.wildcardQuery("content.SPY", "*" + analysisKey + "*").boost(0.8f);
            pingYinSampleContainQueryBuilder = QueryBuilders.boolQuery().should(pingYinTitleSampleContainQueryBuilder).should(pingYinContentSampleContainQueryBuilder);
        }

        /**
         * 拼音全拼搜索
         * 1、分析key，获取拼音词干   case :  南京东路==>[nan,jing,dong,lu]，南京donglu==>[nan,jing,dong,lu]
         * 2、搜索查询，必须匹配所有拼音词，如南京东路，则nan,jing,dong,lu四个词干必须完全匹配
         * 3、如果有中文前缀，则排序优先
         * 权重*1.0
         */
        QueryBuilder pingYinFullQueryBuilder = null;
        if (key.length() > 1) {
            QueryBuilder pingYinTitleFullContainQueryBuilder = QueryBuilders.matchPhraseQuery("title.FPY", ChineseToPinYinUtil.ToPinyin(key)).analyzer("pinyiFullSearchAnalyzer").boost(1.5f);
            QueryBuilder pingYinContentFullContainQueryBuilder = QueryBuilders.matchPhraseQuery("content.FPY", ChineseToPinYinUtil.ToPinyin(key)).analyzer("pinyiFullSearchAnalyzer").boost(1);
            pingYinFullQueryBuilder = QueryBuilders.boolQuery().should(pingYinTitleFullContainQueryBuilder).should(pingYinContentFullContainQueryBuilder);
        }

        /**
         * 完整包含关键字查询(优先级次低，只有以上四种方式查询无结果时才考虑）
         * 权重*0.8
         */
        QueryBuilder containSearchBuilder = multiMatchQuery(key, "title", "content").analyzer("ikSearchAnalyzer").minimumShouldMatch("100%");

        /**
         * 完整包含关键字查询(优先级最低，只有以上五种方式查询无结果时才考虑）
         * 权重*0.5
         */
        QueryBuilder likeSearchBuilder = multiMatchQuery(key, "title", "content").analyzer("ikSearchAnalyzer").boost(0.5f);

        disMaxQueryBuilder
                .add(normSearchBuilder)
//                .add(pingYinSampleQueryBuilder)
                .add(containSearchBuilder)
                .add(likeSearchBuilder);

        //以下两个对性能有一定的影响，故作此判定，单个字符不执行此类搜索
        if (pingYinFullQueryBuilder != null) {
            disMaxQueryBuilder.add(pingYinFullQueryBuilder);
        }
        if (pingYinSampleContainQueryBuilder != null) {
            disMaxQueryBuilder.add(pingYinSampleContainQueryBuilder);
        }

        QueryBuilder queryBuilder = disMaxQueryBuilder;

        //关键如果有中文，则必须包含在内容中（这里把英文的过滤掉）
        if (StringUtils.isNotBlank(startChineseString)) {
            queryBuilder = QueryBuilders.boolQuery()
                    .must(disMaxQueryBuilder)
                    .filter(QueryBuilders.queryStringQuery("*" + startChineseString + "*").field("title.ngram").analyzer("ngramSearchAnalyzer"));

            FunctionScoreQueryBuilder.FilterFunctionBuilder[] filterFunctionBuilders = {
                    new FunctionScoreQueryBuilder.FilterFunctionBuilder(
                            multiMatchQuery(startChineseString, "title.ngram", "content.ngram").analyzer("ngramSearchAnalyzer"),
                            ScoreFunctionBuilders.weightFactorFunction(1.5f)
                    )
            };

            queryBuilder = QueryBuilders.functionScoreQuery(queryBuilder, filterFunctionBuilders);

        }

        logger.debug("新闻搜索[中英文混合搜索]");

        return queryBuilder;
    }

    @Override
    public List<NewsDetail> listNewsDetail(List<Long> ids) {
        BoolQueryBuilder boolQueryBuilder = boolQuery().must(termsQuery("id", ids));
        SearchQuery searchQuery = new NativeSearchQueryBuilder().withQuery(boolQueryBuilder).build();
        logger.debug("新闻查询详情--->QueryDSL:" + searchQuery.getQuery().toString());
        return elasticsearchOperations.queryForList(searchQuery, NewsDetail.class);
    }

    @Override
    public List<News> listNews(List<Long> ids) {
        Pageable pageable = PageRequest.of(0, ids.size());
        BoolQueryBuilder boolQueryBuilder = boolQuery().filter(termsQuery("id", ids));
        SearchQuery searchQuery = new NativeSearchQueryBuilder().withQuery(boolQueryBuilder).withPageable(pageable).build();
        logger.debug("新闻按id列表查询--->QueryDSL:" + searchQuery.getQuery().toString());
        return elasticsearchOperations.queryForList(searchQuery, News.class);
    }

    @Async
    @Override
    public void save(NewsParam newsParam) {
        KeyGenerator keyGenerator = RedisConfig.NEWS_ES_QUEUE.copy().setKey("save");
        try {
            News news = convertNews(newsParam);
//            newsRepository.save(news);
        } catch (Exception e) {
            logger.error("es新闻新增失败,id为:{}", newsParam.getId());
            redisListAdapter.leftPush(keyGenerator, newsParam.getId());
        }
    }

    @Async
    @Override
    public void batchDelNews(List<Long> ids) {
        logger.error("进入删除新闻流程,参数为:{}", ids);
        KeyGenerator keyGenerator = RedisConfig.NEWS_ES_QUEUE.copy().setKey("delete");
        try {
//            ids.forEach(x -> newsRepository.deleteById(String.valueOf(x)));
        } catch (Exception e) {
            logger.error("es新闻删除失败,ids为:{}", ids);
            redisListAdapter.leftPush(keyGenerator, ids);
        }
    }

    @Async
    @Override
    public void batchSave(List<NewsParam> newsParam) {
        logger.warn("进入批量新增方法");
        bulkUpdate(newsParam, "localnews");
    }

    @Override
    public List<OldNews> listOldNewsDetail(List<Long> ids) {
        Pageable pageable = PageRequest.of(0, ids.size());
        BoolQueryBuilder boolQueryBuilder = boolQuery().must(termsQuery("id", ids));
        SearchQuery searchQuery = new NativeSearchQueryBuilder().withQuery(boolQueryBuilder).withPageable(pageable).build();
        logger.debug("新闻按id列表查询--->QueryDSL:" + searchQuery.getQuery().toString());
        return elasticsearchOperations.queryForList(searchQuery, OldNews.class);
    }

    @Override
    public List<News> listTopNews(int size,String areaCode) {
        Pageable pageable = PageRequest.of(0, size);
        BoolQueryBuilder boolQueryBuilder = boolQuery()
                .must(termQuery("status", 1))
                .must(termQuery("top",2))
                .must(termQuery("review_status",2))
                .must(existsQuery("top_expire_time"))
                .must(rangeQuery("top_expire_time").gte(System.currentTimeMillis()));

        if (StringUtils.isNotEmpty(areaCode)) {
            boolQueryBuilder.must(termQuery("area_detail",areaCode));
        }else{
            boolQueryBuilder.mustNot(existsQuery("area_detail"));
        }
        SearchQuery searchQuery = new NativeSearchQueryBuilder()
                .withQuery(boolQueryBuilder).withPageable(pageable)
                .withSort(SortBuilders.fieldSort("issue_time").order(SortOrder.DESC))
                .build();
        logger.debug("新闻查找置顶新闻列表--->QueryDSL:" + searchQuery.getQuery().toString());
        return elasticsearchOperations.queryForList(searchQuery, News.class);
    }

    /**
     * 对象转换
     *
     * @param newsParam
     * @return
     */
    private News convertNews(NewsParam newsParam) {
        News news = new News();
//        news.setContent(newsParam.getContent());
        news.setId(String.valueOf(newsParam.getId()));
        news.setTitle(newsParam.getTitle());
        if (newsParam.getIssueTime() != null) {
//            news.setIssue_time(newsParam.getIssue_time().getTime());
        }
        return news;
    }

    /**
     * 批量新增数据
     *
     * @param newsParamList
     * @param indexName
     * @param indexType
     * @return
     * @throws Exception
     */
    private long bulkIndex(List<NewsParam> newsParamList, String indexName, String indexType) {
        int counter = 0;
        try {
            // 判断索引是否存在
            if (!elasticsearchOperations.indexExists(indexName)) {
                elasticsearchOperations.createIndex(indexName);
            }
            elasticsearchOperations.putMapping(News.class);
            List<IndexQuery> queries = new ArrayList<>();

            for (NewsParam jwrk : newsParamList) {
                IndexQuery indexQuery = new IndexQuery();
                indexQuery.setId(jwrk.getId().toString());
                indexQuery.setSource(JSON.toJSONString(jwrk));
                indexQuery.setIndexName(indexName);
                indexQuery.setType(indexType);
                queries.add(indexQuery);

                // 分批提交索引
                if (counter % 500 == 0) {
                    elasticsearchOperations.bulkIndex(queries);
                    queries.clear();
                    logger.debug("bulkIndex counter : " + counter);
                }
                counter++;
            }

            // 不足批的索引最后不要忘记提交
            if (queries.size() > 0) {
                elasticsearchOperations.bulkIndex(queries);
            }
            elasticsearchOperations.refresh(indexName);
            System.out.println("bulkIndex completed.");
        } catch (Exception e) {
            System.out.println("IndexerService.bulkIndex e;" + e.getMessage());
            throw e;
        }
        return -1;
    }

    /**
     * 批量新增或更新数据，若没有id则新增，若有id则更新数据
     *
     * @param newsDataList
     * @param indexName
     * @return
     * @throws Exception
     */
    private long bulkUpdate(List<NewsParam> newsDataList, String indexName) {
        int counter = 0;
        try {
            elasticsearchOperations.putMapping(News.class);
            List<UpdateQuery> queries = new ArrayList<>();

            long time = System.currentTimeMillis();
            for (NewsParam news : newsDataList) {
                //indexRequest方法如果是对象会报错，所以转为map
                Map<String, Object> map = JSONObject.parseObject(JSON.toJSONString(news));
                IndexRequest indexRequest = new IndexRequest();
                indexRequest.source(map);
                UpdateQuery updateQuery = new UpdateQueryBuilder().withId(String.valueOf(news.getId()))
                        .withDoUpsert(true).withClass(News.class)
                        .withIndexRequest(indexRequest).build();
                queries.add(updateQuery);
                counter++;
                // 分批提交索引
                if (counter != 0 && counter % 5000 == 0) {
                    elasticsearchOperations.bulkUpdate(queries);
                    queries.clear();
                    logger.warn("bulkUpdate counter : " + counter);
                }

            }

            // 不足批的索引最后不要忘记提交
            if (queries.size() > 0) {
                elasticsearchOperations.bulkUpdate(queries);
            }
            elasticsearchOperations.refresh(indexName);
            logger.warn("bulkUpdate completed.Time is : " + (System.currentTimeMillis() - time));
        } catch (Exception e) {
            logger.debug("IndexerService.bulkIndex e;" + e.getMessage());
            throw e;
        }
        return -1;
    }

    /**
     * 根据id批量删除数据
     *
     * @param idList
     * @return
     */
    public boolean deleteByIds(List<String> idList) {
        try {
            CriteriaQuery criteriaQuery = new CriteriaQuery(new Criteria());
            criteriaQuery.setIds(idList);
            elasticsearchOperations.delete(criteriaQuery, News.class);
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

}
