package com.bxm.localnews.thirdparty.service.impl;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.thirdparty.constant.AdvertStatusEunm;
import com.bxm.localnews.thirdparty.constant.AdvertTypeEnum;
import com.bxm.localnews.thirdparty.constant.GlobalFlagEnum;
import com.bxm.localnews.thirdparty.domain.AdvertAreaMapper;
import com.bxm.localnews.thirdparty.domain.AdvertMapper;
import com.bxm.localnews.thirdparty.dto.AdvertDTO;
import com.bxm.localnews.thirdparty.service.AdvertService;
import com.bxm.localnews.thirdparty.vo.Advert;
import com.bxm.localnews.thirdparty.vo.AdvertArea;
import com.bxm.localnews.thirdparty.vo.AdvertRecord;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * Created by hsq 10:33 2018/2/8
 */
@Service
public class AdvertServiceImpl implements AdvertService {

    private static final int AREA_CODE_LENGTH = 12;

    private AdvertMapper advertMapper;

    private AdvertAreaMapper advertAreaMapper;

    private RedisStringAdapter redisStringAdapter;

    private LocationFacadeService locationFacadeService;

    @Autowired
    public AdvertServiceImpl(AdvertMapper advertMapper, AdvertAreaMapper advertAreaMapper,
                             RedisStringAdapter redisStringAdapter, LocationFacadeService locationFacadeService) {
        this.advertMapper = advertMapper;
        this.advertAreaMapper = advertAreaMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.locationFacadeService = locationFacadeService;
    }

    /**
     * 获取对应类型的所有广告
     *
     * @return
     */
    private List<Advert> getAllAdvertByType(Byte type) {
        TypeReference<List<Advert>> typeReference = new TypeReference<List<Advert>>() {
        };
        List<Advert> adverts = this.redisStringAdapter.get(this.getAdvertListKey(type), typeReference);

        if (CollectionUtils.isEmpty(adverts)) {
            adverts = this.advertMapper.queryAdByType(type, AdvertStatusEunm.PUTAWAY.getType());
            if (CollectionUtils.isNotEmpty(adverts)) {
                //保存一天
                this.redisStringAdapter.set(this.getAdvertListKey(type), adverts);
            }
        }

        return adverts;
    }

    @Override
    public List<AdvertDTO> queryAdByType(Byte type, String areaCode, Long userId) {
        List<Long> longList = Lists.newArrayList();

        List<Advert> advertList = getAllAdvertByType(type);
        if (StringUtils.isNotBlank(areaCode)) {
            longList = getAdvertAreasByAreaCode(areaCode);
        }

        List<AdvertDTO> advertDTOList = getAdvertDTOs(advertList, longList);

        //根据时间过滤
        advertDTOList = advertDTOList.stream().filter(advertDTO -> removeByPutTime(advertDTO)).collect(Collectors.toList());


        //首页弹窗广告-------根据用户信息判断是否返回广告信息
        if (AdvertTypeEnum.SHOUYE_TANCH.getType().equals(String.valueOf(type))) {
            List<AdvertDTO> advertDTOS = filterAdvertInfo(userId, advertDTOList);

            return advertDTOS;
        }

        //记录广告查看信息
        if (userId != null && CollectionUtils.isNotEmpty(advertDTOList)) {
            recordAdvert(userId, advertDTOList);
        }

        return advertDTOList;
    }

    /**
     * 判断时间是否处于投放开始时间和结束时间之间
     *
     * @param advertDTO
     * @return
     */
    private Boolean removeByPutTime(AdvertDTO advertDTO) {
        Date now = new Date();
        return !(advertDTO.getStartTime() != null && DateUtils.after(advertDTO.getStartTime(), now)
                || (advertDTO.getEndTime() != null && DateUtils.before(advertDTO.getEndTime(), now)));
    }

    /**
     * 记录广告查看时间
     *
     * @param userId
     * @param advertDTOList
     */
    private void recordAdvert(Long userId, List<AdvertDTO> advertDTOList) {
        List<AdvertRecord> advertRecords = Lists.newArrayList();
        Date now = new Date();
        advertDTOList.forEach(advertDTO -> {
            AdvertRecord advertRecord = new AdvertRecord();
            advertRecord.setAdvertId(advertDTO.getId());
            advertRecord.setUserId(userId);
            advertRecord.setViewTime(now);
            advertRecords.add(advertRecord);
        });
        if (CollectionUtils.isEmpty(advertRecords)) {
            return;
        }

        this.advertMapper.recordAdvert(advertRecords);
    }

    @Override
    public Advert selectByPrimaryKey(Long id) {
        return this.advertMapper.selectByPrimaryKey(id);
    }

    @Override
    public List<AdvertDTO> getListAds(int size) {
        return this.advertMapper.getListAds(size);
    }

    /**
     * 通过广告列表和广告id列表返回对应的广告信息
     *
     * @param advertList
     * @param longList
     * @return
     */
    private List<AdvertDTO> getAdvertDTOs(List<Advert> advertList, List<Long> longList) {
        List<AdvertDTO> advertDTOList = Lists.newArrayList();

        if (CollectionUtils.isNotEmpty(advertList)) {
            for (Advert advert : advertList) {
                if (GlobalFlagEnum.IS_NOT_GLOBAL.getState().equals(advert.getGlobalFlag())
                        && ((!longList.contains(advert.getId())))) {
                    continue;
                }

                AdvertDTO advertDTO = getAdvertDTOByAdvert(advert);
                advertDTOList.add(advertDTO);
            }
        }

        return advertDTOList;
    }

    /**
     * 获取广告类列表的key
     *
     * @param type
     * @return
     */
    private KeyGenerator getAdvertListKey(Byte type) {
        return RedisConfig.THIRDPARTY_TYPE_ADVERT.copy().appendKey(type).appendKey("list");
    }

    /**
     * 类型转换
     *
     * @param advert
     * @return
     */
    private AdvertDTO getAdvertDTOByAdvert(Advert advert) {
        AdvertDTO advertDTO = new AdvertDTO();
        BeanUtils.copyProperties(advert, advertDTO);

        return advertDTO;
    }

    /**
     * 通过地区编码获取广告id列表
     *
     * @param areaCode
     * @return
     */
    private List<Long> getAdvertAreasByAreaCode(String areaCode) {
        areaCode = complteAreaCode(areaCode);
        return this.advertAreaMapper.getAllAdvertAreaByAreaCode(areaCode)
                .stream()
                .map(AdvertArea::getAdvertId)
                .collect(Collectors.toList());
    }

    /**
     * 地区编码转换成12位的编码
     *
     * @param areaCode
     * @return
     */
    private String complteAreaCode(String areaCode) {
        String fullCode = areaCode;
        if (AREA_CODE_LENGTH != fullCode.length()) {
            //当定位时取得的是6位城市编码
            fullCode = areaCode + "000000";
            com.bxm.localnews.base.dto.LocationDTO locationDTO = locationFacadeService.getLocationByCode(fullCode);
            //如果是区是查不到的，查询上一级
            if (null == locationDTO) {
                fullCode = areaCode.substring(0, 4) + "00000000";
            }
        }
        return fullCode;
    }

    /**
     * 检测是否返回广告信息
     *
     * @param userId 用户id
     * @param list   广告列表
     * @return
     */
    private List<AdvertDTO> filterAdvertInfo(Long userId, List<AdvertDTO> list) {
        List<AdvertDTO> advertDTOS = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return advertDTOS;
        }

        advertDTOS = list.stream().filter(advertDTO ->
                !checkRemoveAdvertInfo(userId, advertDTO.getId(), advertDTO.getShowType()))
                .collect(Collectors.toList());
        return advertDTOS;
    }

    /**
     * 是否移除广告当天剩余
     *
     * @return
     */
    private Boolean checkRemoveAdvertInfo(Long userId, Long advertId, Byte showType) {
        if (redisStringAdapter.hasKey(getAdvertByTypeKey(showType, userId, advertId))) {
            return true;
        }
        if (showType != null) {
            if (showType == 3) {
                redisStringAdapter.set(getAdvertByTypeKey(showType, userId, advertId), userId, DateUtils.getCurSeconds());
            } else {
                redisStringAdapter.set(getAdvertByTypeKey(showType, userId, advertId), userId);
            }
        }
        return false;
    }

    /**
     * 广告对应用户展示方式
     *
     * @param showType
     * @param userId
     * @return
     */
    private KeyGenerator getAdvertByTypeKey(Byte showType, Long userId, Long advertId) {
        return RedisConfig.THIRDPARTY_SHOW_TYPE_ADVERT.copy().appendKey(advertId + "_" + showType + "_" + userId);
    }
}
