package com.bxm.localnews.thirdparty.service.impl;

import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.thirdparty.constant.GlobalFlagEnum;
import com.bxm.localnews.thirdparty.constant.ToolsSensitiveFlagEnum;
import com.bxm.localnews.thirdparty.constant.ToolsStatusEnum;
import com.bxm.localnews.thirdparty.domain.ToolsMapper;
import com.bxm.localnews.thirdparty.dto.ToolsDTO;
import com.bxm.localnews.thirdparty.service.ToolsService;
import com.bxm.localnews.thirdparty.vo.Tools;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author zhaoyadong 2018/10/19 13:48
 * @desc
 */
@Service
public class ToolsServiceImpl implements ToolsService {

    private final static List<String> replaceList = Lists.newArrayList("{uid}");

    private static final int AREA_CODE_LENGTH = 12;

    private ToolsMapper toolsMapper;

    private RedisHashMapAdapter redisHashMapAdapter;

    private AppVersionSupplyService appVersionSupplyService;

    private LocationFacadeService locationFacadeService;

    @Autowired
    public ToolsServiceImpl(ToolsMapper toolsMapper, RedisHashMapAdapter redisHashMapAdapter,
                            AppVersionSupplyService appVersionSupplyService, LocationFacadeService locationFacadeService) {
        this.toolsMapper = toolsMapper;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.appVersionSupplyService = appVersionSupplyService;
        this.locationFacadeService = locationFacadeService;
    }

    @Override
    public List<ToolsDTO> getToolsByAreaCode(String areaCode, Long userId, Integer isLogin, BasicParam basicParam,
                                             Byte locationType) {
        Boolean isPublishState = appVersionSupplyService.getPublishState(basicParam);

        List<ToolsDTO> toolsDTOS = getToolDTOsByToolsList(areaCode, isPublishState, locationType);

        //设置对应替换的参数
        List<String> list = Lists.newArrayList();
        if (isLogin == 1) {
            list.add(userId.toString());
        } else {
            list.add("");
        }

        toolsDTOS.forEach(t -> {
            for (int i = 0; i < replaceList.size(); i++) {
                String str = replaceList.get(i);
                if (t.getUrl().contains(replaceList.get(i))) {
                    t.setUrl(t.getUrl().replace(str, list.get(i)));
                }
            }
        });

        return toolsDTOS;
    }

    /**
     * 获取所有的工具类
     * @param areaCode
     * @return
     */
    private List<Tools> getAllTools(String areaCode) {
        TypeReference<List<Tools>> typeReference = new TypeReference<List<Tools>>() {
        };
        List<Tools> toolsList = this.redisHashMapAdapter.get(this.getToolsListKey(), areaCode, typeReference);

        if (CollectionUtils.isEmpty(toolsList)) {
            String fullAreaCode = complteAreaCode(areaCode);
            toolsList = this.toolsMapper.getAllToolsByStatusAndAreaCode(ToolsStatusEnum.ENABLE.getStatus(), fullAreaCode,
                    GlobalFlagEnum.IS_GLOBAL.getState());
            if (CollectionUtils.isNotEmpty(toolsList)) {
                this.redisHashMapAdapter.put(this.getToolsListKey(), areaCode, toolsList);
            }
        }

        return toolsList;
    }

    /**
     * 获取工具类列表的key
     * @return
     */
    private KeyGenerator getToolsListKey() {
        return RedisConfig.THIRDPARTY_AREACODE_TOOLS.copy().appendKey("list");
    }

    /**
     * 类型转换
     * @param tools
     * @return
     */
    private ToolsDTO getToolsDTOByTools(Tools tools) {
        ToolsDTO toolsDTO = new ToolsDTO();
        BeanUtils.copyProperties(tools, toolsDTO);
        return toolsDTO;
    }

    /**
     * 通过地区编码、提包状态获取工具类列表
     * @param areaCode
     * @param isPublishState
     * @return
     */
    private List<ToolsDTO> getToolDTOsByToolsList(String areaCode, Boolean isPublishState, Byte locationType) {

        List<Tools> toolsList = getAllTools(areaCode);
        if (locationType != null && locationType == 1) {
            toolsList = toolsList.stream().filter(tools ->
                    (tools.getHomeFlag() != null && tools.getHomeFlag() == 1)
            ).collect(Collectors.toList());
            toolsList = getTop5ForList(toolsList);
        } else {
            toolsList = toolsList.stream().filter(tools ->
                    (tools.getHomeFlag() == null || tools.getHomeFlag() == 0)
            ).collect(Collectors.toList());
        }
        List<ToolsDTO> toolsDTOS = toolsList.stream().filter(tools -> !(isPublishState &&
                ToolsSensitiveFlagEnum.IS_SENSITIVE.getState().equals(tools.getSensitiveFlag())))
                .map(this::getToolsDTOByTools)
                .collect(Collectors.toList());

        return toolsDTOS;
    }

    private List<Tools> getTop5ForList(List<Tools> toolsList) {
        if (toolsList.size() > 5) {
            List<Tools> tools = Lists.newArrayList();
            for (int i = 0; i < 5; i++) {
                tools.add(toolsList.get(i));
            }
            return tools;
        }

        return toolsList;
    }

    /**
     * 地区编码转换成12位的编码
     * @param areaCode
     * @return
     */
    private String complteAreaCode(String areaCode) {
        String fullCode = areaCode;
        if (StringUtils.isNotBlank(fullCode)) {
            if (AREA_CODE_LENGTH != fullCode.length()) {
                //当定位时取得的是6位城市编码
                fullCode = areaCode + "000000";
                com.bxm.localnews.base.dto.LocationDTO locationDTO = locationFacadeService.getLocationByCode(fullCode);
                //如果是区是查不到的，查询上一级
                if (null == locationDTO) {
                    fullCode = areaCode.substring(0, 4) + "00000000";
                }
            }
        }

        return fullCode;
    }
}
