package com.bxm.localnews.thirdparty.filter;

import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.dto.UserRegisterInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.thirdparty.constant.AdvertShowTypeEnum;
import com.bxm.localnews.thirdparty.param.AdvertisementParam;
import com.bxm.localnews.thirdparty.vo.AdvertVO;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.BitOperatorUtil;
import com.bxm.newidea.component.tools.DateUtils;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum.JOINED_LOTTERY;
import static org.apache.commons.lang3.time.DateUtils.isSameDay;

/**
 * @Author: pf.w
 * @Date: 2020/5/13 9:28
 */
@Log4j2
public abstract class AbstractAdvertFilterHandler {

    @Autowired
    RedisStringAdapter redisStringAdapter;

    @Autowired
    private UserIntegrationService userIntegrationService;

    public boolean filter(List<AdvertVO> advertVOS, AdvertisementParam param) {
        if (null == param) {
            return false;
        }
        return doFilter(advertVOS, param);
    }

    /**
     * 执行具体的过滤器逻辑
     *
     * @param advertVOS 当前广告位对应的所有广告
     * @param param     原始请求参数
     * @return true表示继续处理，false表示中断过滤链
     */
    protected abstract boolean doFilter(List<AdvertVO> advertVOS, AdvertisementParam param);


    /**
     * 是否移除广告当天剩余
     *
     * @return
     */
    protected Boolean checkRemoveAdvertInfo(Long userId, Long advertId, Byte showType) {

        if (Boolean.TRUE.equals(redisStringAdapter.hasKey(getAdvertByTypeKey(showType, userId, advertId)))) {
            return true;
        }
        if (showType != null) {
            if (showType == AdvertShowTypeEnum.DAY_FIRST_OPEN_APP_SHOW.getCode()) {
                // 每天第一次打开App时展示
                redisStringAdapter.set(getAdvertByTypeKey(showType, userId, advertId), userId, DateUtils.getCurSeconds());
            } else if (showType == AdvertShowTypeEnum.NEW_USER_SHOW_ONCE.getCode()) {
                // 先设置缓存
                redisStringAdapter.set(getAdvertByTypeKey(showType, userId, advertId), userId);
                // 判断是否新用户
                UserRegisterInfoDTO userRegisterInfoDTO = userIntegrationService.userRegisterInfo(userId);

                // 非新用户返回true过滤 否则新用户返回false 展示一次
                return userRegisterInfoDTO.getFirstLoginTime() == null
                        || !isSameDay(userRegisterInfoDTO.getFirstLoginTime(), new Date());
            } else if (showType == AdvertShowTypeEnum.NOT_JOIN_LOTTERY_OLD_USER.getCode()) {
                // 未参与夺宝老用户展示一次
                redisStringAdapter.set(getAdvertByTypeKey(showType, userId, advertId), userId);

                // 判断是否新用户
                UserRegisterInfoDTO userRegisterInfoDTO = userIntegrationService.userRegisterInfo(userId);
                if (userRegisterInfoDTO.getFirstLoginTime() == null
                        || isSameDay(userRegisterInfoDTO.getFirstLoginTime(), new Date())) {
                    // 新用户返回true过滤
                    return true;
                }

                // 老用户 判断是否参与过夺宝
                UserInfoDTO userFromRedisDB = userIntegrationService.getUserFromRedisDB(userId);
                // 空 过滤
                return Objects.isNull(userFromRedisDB)
                        // 已参与 过滤
                        || BitOperatorUtil.getBitAsBoolean(userFromRedisDB.getJudgeMarker(), JOINED_LOTTERY.getIndex());
            } else {
                // 所有用户展示一次、默认展示
                redisStringAdapter.set(getAdvertByTypeKey(showType, userId, advertId), userId);
            }
        }
        return false;
    }

    private KeyGenerator getAdvertByTypeKey(Byte showType, Long userId, Long advertId) {
        return RedisConfig.THIRDPARTY_SHOW_TYPE_ADVERT.copy().appendKey(advertId + "_" + showType + "_" + userId);
    }
}
