package com.bxm.localnews.thirdparty.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.activity.service.RedPacketFacadeService;
import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.base.service.AreaWhiteBlackService;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.constant.AreaWhiteBlackKeyEnum;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.thirdparty.config.TooltipProperties;
import com.bxm.localnews.thirdparty.constant.ToolsSensitiveFlagEnum;
import com.bxm.localnews.thirdparty.domain.ToolsLocationMapper;
import com.bxm.localnews.thirdparty.domain.ToolsMapper;
import com.bxm.localnews.thirdparty.dto.ToolsDTO;
import com.bxm.localnews.thirdparty.service.ToolsService;
import com.bxm.localnews.thirdparty.vo.Tools;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author zhaoyadong 2018/10/19 13:48
 * @desc
 */
@Service
public class ToolsServiceImpl implements ToolsService {
    private static final Logger LOG = LoggerFactory.getLogger(ToolsServiceImpl.class);
    private static final List<String> REPLACE_LIST = Lists.newArrayList("{uid}");

    private ToolsMapper toolsMapper;

    /**
     * tool location 数据接口层
     */
    private ToolsLocationMapper toolsLocationMapper;

    private AppVersionSupplyService appVersionSupplyService;

    private LocationFacadeService locationFacadeService;

    /**
     * 地区黑白名单服务
     */
    private AreaWhiteBlackService areaWhiteBlackService;

    private RedPacketFacadeService redPacketFacadeService;

    private TooltipProperties tooltipProperties;

    @Autowired
    public ToolsServiceImpl(ToolsMapper toolsMapper,
                            ToolsLocationMapper toolsLocationMapper,
                            AppVersionSupplyService appVersionSupplyService,
                            LocationFacadeService locationFacadeService,
                            AreaWhiteBlackService areaWhiteBlackService,
                            RedPacketFacadeService redPacketFacadeService,
                            TooltipProperties tooltipProperties) {
        this.toolsMapper = toolsMapper;
        this.toolsLocationMapper = toolsLocationMapper;
        this.appVersionSupplyService = appVersionSupplyService;
        this.locationFacadeService = locationFacadeService;
        this.areaWhiteBlackService = areaWhiteBlackService;
        this.redPacketFacadeService = redPacketFacadeService;
        this.tooltipProperties = tooltipProperties;
    }

    @Override
    public List<ToolsDTO> getToolsByAreaCode(String areaCode, Long userId, Integer isLogin, BasicParam basicParam,
                                             Byte locationType) {
        Boolean isPublishState = appVersionSupplyService.getPublishState(basicParam);

        List<ToolsDTO> toolsDTOS = getToolDTOsByToolsList(areaCode, isPublishState, locationType, userId, basicParam.getCurVer());

        //设置对应替换的参数
        List<String> list = Lists.newArrayList();
        if (isLogin == 1) {
            list.add(userId.toString());
        } else {
            list.add("");
        }

        toolsDTOS.forEach(t -> {
            for (int i = 0; i < REPLACE_LIST.size(); i++) {
                String str = REPLACE_LIST.get(i);
                if (t.getUrl().contains(REPLACE_LIST.get(i))) {
                    t.setUrl(t.getUrl().replace(str, list.get(i)));
                }
            }
        });

        return toolsDTOS;
    }

    /**
     * 获取所有的工具类
     *
     * @param areaCode 地区code
     * @return 获取所有的工具类
     */
    private List<Tools> getRealTools(String areaCode) {
        //适配地区code
        String fullAreaCode = locationFacadeService.completeAreaCode(areaCode);

        //先从缓存中取
        List<Tools> cacheToolsList = this.areaWhiteBlackService.getCacheByAreaCode(
                AreaWhiteBlackKeyEnum.TOOLS.getKey(), fullAreaCode, new TypeReference<List<Tools>>() {
                });
        if (CollectionUtils.isNotEmpty(cacheToolsList)) {
            return cacheToolsList;
        }

        //获取所有上线状态的工具
        List<Tools> allToolsList = this.toolsMapper.getAllTools();
        //获取白名单地区列表
        List<Long> whiteAreaList = toolsLocationMapper.getToolLocationByType(areaCode, 1);
        //获取黑名单地区列表
        List<Long> blackAreaList = toolsLocationMapper.getToolLocationByType(areaCode, 2);

        List<Tools> realToolsList = this.areaWhiteBlackService.filterCacheByAreaCode(allToolsList, whiteAreaList, blackAreaList);
        this.areaWhiteBlackService.setCacheByAreaCode(AreaWhiteBlackKeyEnum.TOOLS.getKey(), fullAreaCode, realToolsList);
        LOG.info("根据地区{}获取得到的工具列表信息为:{}", areaCode, JSON.toJSONString(realToolsList));

        return realToolsList;
    }

    /**
     * 类型转换
     *
     * @param tools
     * @return
     */
    private ToolsDTO getToolsDTOByTools(Tools tools) {
        ToolsDTO toolsDTO = new ToolsDTO();
        BeanUtils.copyProperties(tools, toolsDTO);
        return toolsDTO;
    }

    /**
     * 通过地区编码、提包状态获取工具类列表
     *
     * @param areaCode       地区code
     * @param isPublishState 提包状态
     * @param userId         用户id
     * @return
     */
    private List<ToolsDTO> getToolDTOsByToolsList(String areaCode, Boolean isPublishState, Byte locationType,
                                                  Long userId, String curVer) {

        //获取该区域所有符合要求的工具
        List<Tools> toolsList = getRealTools(areaCode);

        //根据策略信息完善工具列表-工具提示语
        getTooltipByStrategy(toolsList, userId, areaCode);

        if (locationType != null && locationType == 1) {
            toolsList = toolsList.stream().filter(tools ->
                    (tools.getHomeFlag() != null && tools.getHomeFlag() == 1)
            ).collect(Collectors.toList());
            //2.1.0版本不需要截取
//            toolsList = getTop5ForList(toolsList);
        } else {
            toolsList = toolsList.stream().filter(tools ->
                    (tools.getHomeFlag() == null || tools.getHomeFlag() == 0)
            ).collect(Collectors.toList());
        }

        List<ToolsDTO> toolsDTOS = toolsList.stream().filter(tools -> !(isPublishState &&
                ToolsSensitiveFlagEnum.IS_SENSITIVE.getState().equals(tools.getSensitiveFlag())))
                .map(this::getToolsDTOByTools)
                .collect(Collectors.toList());

        return toolsDTOS;
    }

    private List<Tools> getTop5ForList(List<Tools> toolsList) {
        if (toolsList.size() > 5) {
            List<Tools> tools = Lists.newArrayList();
            for (int i = 0; i < 5; i++) {
                tools.add(toolsList.get(i));
            }
            return tools;
        }

        return toolsList;
    }

    /**
     * 获取工具提示语
     *
     * @param toolsList 工具信息list
     * @param userId    用户id
     * @param areaCode  地区编码
     */
    private void getTooltipByStrategy(List<Tools> toolsList, Long userId, String areaCode) {
        for (Tools toolInfo : toolsList) {
            if ("RED_PACKET".equals(toolInfo.getTooltip())) {
                if (redPacketFacadeService.getHasAvailableRedPacket(userId, areaCode)) {
                    toolInfo.setTooltip(tooltipProperties.getRedPacket());
                } else {
                    toolInfo.setTooltip(null);
                }
            }
        }
    }

}
