package com.bxm.localnews.thirdparty.service.impl;

import com.bxm.localnews.base.service.BaseUrlFacadeService;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.common.param.GetAvailableDomainInfoParam;
import com.bxm.localnews.common.vo.BaseUrlInfo;
import com.bxm.localnews.common.vo.ViewSceneInfo;
import com.bxm.localnews.thirdparty.config.WechatAccountConfig;
import com.bxm.localnews.thirdparty.config.WechatMPAuthConfig;
import com.bxm.localnews.thirdparty.config.WechatMPConfig;
import com.bxm.localnews.thirdparty.param.RedirectToWechatMpAuthParam;
import com.bxm.localnews.thirdparty.service.WechatMpFacadeService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.bean.WxJsapiSignature;
import me.chanjar.weixin.common.util.http.URIUtil;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.util.WxMpConfigStorageHolder;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.net.URL;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static com.bxm.localnews.common.constant.DomainScene.DomainViewScene.WX_JS_VIEW;
import static com.gexin.fastjson.JSON.toJSON;
import static com.gexin.fastjson.JSON.toJSONString;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author gonzo
 * @date 2020-08-03 15:21
 **/
@Slf4j
@Service
@AllArgsConstructor
public class WechatMpServiceImpl implements WechatMpService, WechatMpFacadeService {

    private final WxMpService wxMpService;

    private final BaseUrlFacadeService baseUrlFacadeService;

    private final WechatMPConfig wechatMPConfig;

    private final WechatMPAuthConfig wechatMPAuthConfig;

    private final WechatAccountConfig wechatAccountConfig;

    @Override
    public String wechatMpAuthUrlGen(RedirectToWechatMpAuthParam param) {
        if (isBlank(param.getScene())) {
            log.warn("获取微信授权/支付重定向url失败，场景为空，param: {}", toJSONString(param));
            return param.getRedirectUrl();
        }

        // 根据场景获取域名
        GetAvailableDomainInfoParam infoParam = new GetAvailableDomainInfoParam();
        infoParam.setScene(param.getScene());
        if (Objects.equals(Objects.toString(DomainScene.WECHAT_PAY), param.getScene())) {
            infoParam.setAppId(wechatMPConfig.getActive().getActiveAppId());
        } else {
            // 支付用支付的
            infoParam.setAppId(wechatAccountConfig.getJsapiAppId());
        }

        BaseUrlInfo domainInfo = baseUrlFacadeService.getBaseUrlInfo(infoParam);

        if (Objects.isNull(domainInfo) || isBlank(domainInfo.getBaseUrl())) {
            log.warn("根据场景: {} 获取域名失败，param: {}, infoParam: {}", param.getScene(), toJSONString(param),
                    toJSONString(infoParam));
            return param.getRedirectUrl();
        }

        // 根据类型 获取中间页的url
        String urlForScene = wechatMPAuthConfig.getAuthPath().get(param.getScene());
        // 拼接中间页域名 得到中间页url 再带上最终的重定向地址
        String authUrlWithParam = StringUtils.join(domainInfo.getBaseUrl(), urlForScene, "?", "redirectUrl=",
                URIUtil.encodeURIComponent(param.getRedirectUrl()));
        // 组装redirectUrl
        String finalRedirectUrl = wxMpService.oauth2buildAuthorizationUrl(authUrlWithParam, param.getScope(), param.getState());

        if (log.isDebugEnabled()) {
            log.debug("生成的 finalRedirectUrl: {}", finalRedirectUrl);
        }

        return finalRedirectUrl;
    }

    @Override
    public WxJsapiSignature createJsApiSignature(String sourceUrl) {

        // 获取当前域名对应的公众号
        try {
            URL urlInstance = new URL(sourceUrl);

            // 根据域名 获取域名对应的落地页投放信息
            List<ViewSceneInfo> viewSceneByDomain = baseUrlFacadeService.getViewSceneByDomain(urlInstance.getHost());

            // 过滤出微信JS API的投放类型
            String wxJsView = Objects.toString(WX_JS_VIEW);
            Optional<ViewSceneInfo> first = viewSceneByDomain.stream()
                    .filter(p -> Objects.equals(p.getViewScene(), wxJsView) && isNotBlank(p.getAppId()))
                    .findFirst();

            if (!first.isPresent()) {
                log.warn("域名: {} 获取不到对应配置的app id", urlInstance.getHost());
                return null;
            }

            String appId = first.get().getAppId();
            log.info("域名: {} 获取到的appId: {}", urlInstance.getHost(), appId);

            // 设置启用的appId
            WxMpConfigStorageHolder.set(appId);

            // 根据appId和短链进行微信js api授权
            WxJsapiSignature jsapiSignature = wxMpService.createJsapiSignature(sourceUrl);
            if (log.isDebugEnabled()) {
                log.debug("urL {} 获取到的js 验签信息: {} ", sourceUrl, toJSON(jsapiSignature));
            }
            return jsapiSignature;
        } catch(Exception e) {
            log.error("url: {} 不是一个有效的url", sourceUrl, e);
        }

        return null;
    }
}
