package com.bxm.localnews.thirdparty.service.impl;

import cn.binarywang.wx.miniapp.api.WxMaService;
import cn.binarywang.wx.miniapp.bean.WxMaCodeLineColor;
import cn.binarywang.wx.miniapp.bean.WxMaJscode2SessionResult;
import cn.binarywang.wx.miniapp.bean.WxMaPhoneNumberInfo;
import cn.binarywang.wx.miniapp.bean.WxMaUserInfo;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.httpclient.service.HttpClientService;
import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.integration.UserAuthIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.param.UserParam;
import com.bxm.localnews.thirdparty.config.WechatMPConfig;
import com.bxm.localnews.thirdparty.config.WechatProperties;
import com.bxm.localnews.thirdparty.config.WxMaConfiguration;
import com.bxm.localnews.thirdparty.constant.WxMaTypeEnum;
import com.bxm.localnews.thirdparty.dto.WechatDto;
import com.bxm.localnews.thirdparty.dto.WxMaUserData;
import com.bxm.localnews.thirdparty.dto.WxUserInfo;
import com.bxm.localnews.thirdparty.service.WechatService;
import com.bxm.localnews.thirdparty.vo.WechatBackResult;
import com.bxm.localnews.vo.User;
import com.bxm.localnews.vo.UserAuth;
import com.bxm.newidea.component.oss.service.AliyunOSSService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import me.chanjar.weixin.common.error.WxErrorException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.File;
import java.util.Date;
import java.util.Map;
import java.util.UUID;

@Service
@AllArgsConstructor(onConstructor = @__(@Autowired))
public class WechatServiceImpl implements WechatService {
    private static final Logger logger = LoggerFactory.getLogger(WechatServiceImpl.class);
    private static final String BASE_SENSITIVE_WORD_URL = "https://api.weixin.qq.com/wxa/msg_sec_check?access_token=";

    private final RedisStringAdapter redisStringAdapter;

    private final WechatProperties wechatProperties;

    private final HttpClientService httpClientService;

    private final AliyunOSSService aliyunOSSService;

    private final UserIntegrationService userIntegrationService;

    private final UserAuthIntegrationService userAuthIntegrationService;

    private final RedisListAdapter redisListAdapter;

    private final WechatMPConfig wechatMpConfig;


    @Override
    public WxMaService getWxMaService(String appid) {
        if (StringUtils.isBlank(appid)) {
            appid = wechatProperties.getAppId();
        }
        return WxMaConfiguration.getMaService(appid);
    }

    @Override
    public WxUserInfo getWechatInfo(WechatDto wechatDto) {
        WxUserInfo wxUserInfo = new WxUserInfo();
        String code = wechatDto.getCode();
        if (StringUtils.isBlank(code)) {
            return wxUserInfo;
        }

        try {
            WxMaJscode2SessionResult wxMaJscode2SessionResult = getWxMaService(wechatDto.getAppId())
                    .jsCode2SessionInfo(wechatDto.getCode());
            return convertWxMaResult(wxMaJscode2SessionResult, wxUserInfo, wechatDto.getType());
        } catch (WxErrorException e) {
            logger.error("微信小程序【大转盘】根据code获取openid错误, [{}]", e);
            return wxUserInfo;
        }
    }

    /**
     * 转换根据code码获取的返回结果
     *
     * @param wxMaJscode2SessionResult
     * @param wxUserInfo
     * @param type                     小程序类型
     * @return
     */
    private WxUserInfo convertWxMaResult(WxMaJscode2SessionResult wxMaJscode2SessionResult, WxUserInfo wxUserInfo, String type) {
        String openId = wxMaJscode2SessionResult.getOpenid();
        wxUserInfo.setOpenId(openId);
        wxUserInfo.setSessionKey(wxMaJscode2SessionResult.getSessionKey());
        wxUserInfo.setUnionId(wxMaJscode2SessionResult.getUnionid());
        if (StringUtils.isNotBlank(openId)) {
            UserAuth userAuth = userAuthIntegrationService.selectAppletUserAuthByOpenId(openId, WxMaTypeEnum.valueOf(type).getType());
            if (null != userAuth) {
                if (null != userAuth.getUserId()) {
                    wxUserInfo.setUserId(userAuth.getUserId());
                    User user = userIntegrationService.selectByPrimaryKey(userAuth.getUserId());
                    if (user != null) {
                        wxUserInfo.setNickname(user.getNickname());
                        wxUserInfo.setHeadImg(user.getHeadImg());
                        wxUserInfo.setPhone(user.getPhone());
                        wxUserInfo.setLocationCode(user.getLocationCode());
                        wxUserInfo.setLocationName(user.getLocationName());
                        if (StringUtils.isBlank(user.getWeixin())) {
                            wxUserInfo.setAuth(false);
                        }
                    }
                }
            }
        }
        return wxUserInfo;
    }

    @Override
    public String getWechatQRCode(WechatDto wechatDto) {

        WxMaCodeLineColor wxMaCodeLineColor = JSON.parseObject(wechatDto.getLineColor(), WxMaCodeLineColor.class);
        try {
            if (StringUtils.isBlank(wechatDto.getScene())) {
                logger.info("前端未传递scene值，使用默认的1011。参数为：[{}]", wechatDto);
                wechatDto.setScene("1011");
            }
            File file = getWxMaService(wechatDto.getAppId()).getQrcodeService().createWxaCodeUnlimit(wechatDto.getScene(),
                    wechatDto.getPage(), wechatDto.getWidth(), wechatDto.getAutoColor(), wxMaCodeLineColor,
                    wechatDto.getHyaline());
            String imgName = DateUtils.formatDate(new Date()) + "/" + UUID.randomUUID() + "-" +
                    wechatDto.getInviteUserId() + ".png";
            return aliyunOSSService.upload(file, "/qr/" + imgName);
        } catch (WxErrorException e) {
            logger.error("生成小程序二维码错误,请求参数：[{}]", wechatDto);
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    @Override
    public String createUser(WxMaUserData wxMaUserData) {
        logger.info("小程序授权登陆：解析数据：[{}]", JSON.toJSONString(wxMaUserData));
        WxMaUserInfo wxMaUserInfo = getWxMaService(wxMaUserData.getAppId()).getUserService().getUserInfo(wxMaUserData.getSessionKey(),
                wxMaUserData.getEncryptedData(), wxMaUserData.getIv());
        if (wxMaUserInfo != null) {
            UserParam userParam = convertUser(wxMaUserData, wxMaUserInfo);
            logger.info("创建的用户信息：[{}]", JSON.toJSONString(userParam));
            Message message = null;
            if (WxMaTypeEnum.APPLET_DRAW.getName().equals(wxMaUserData.getType())) {
                //抽奖小程序执行以前的逻辑
                logger.info("-------大转盘小程序创建用户---");
                userParam.setInviteType(InviteTypeEnum.APPLETDRAW.getName());

                message = userIntegrationService.createUser(userParam);
            }

            if (WxMaTypeEnum.APPLET_WST.getName().equals(wxMaUserData.getType())) {
                logger.info("-------万事通小程序创建用户---");
                userParam.setInviteType(InviteTypeEnum.MINI_WST.getName());
                message = userIntegrationService.updateUser(userParam);
            }

            if (message != null && message.isSuccess()) {
                logger.info("小程序授权登陆返回信息：[{}]", JSON.toJSONString(message));
                return message.getParamMap().get("userId").toString();
            }
        }

        return "";
    }

    /**
     * 封装用户信息
     *
     * @param wxMaUserData
     * @param wxMaUserInfo
     * @return
     */
    private UserParam convertUser(WxMaUserData wxMaUserData, WxMaUserInfo wxMaUserInfo) {
        UserParam userParam = new UserParam();
        userParam.setOpenId(wxMaUserInfo.getOpenId());
        userParam.setUnionId(wxMaUserInfo.getUnionId());
        userParam.setHeadImg(wxMaUserInfo.getAvatarUrl());
        userParam.setNickName(wxMaUserInfo.getNickName());
        userParam.setSex(Byte.valueOf(wxMaUserInfo.getGender()));

        userParam.setRegIp(wxMaUserData.getIp());
        userParam.setPlatform(wxMaUserData.getPlatform());
        userParam.setRegisteredAddress(wxMaUserData.getArea());
        userParam.setInviteUserId(wxMaUserData.getInviteUserId());
        userParam.setRelationId(wxMaUserData.getRelationId());
        userParam.setTaskInvited(wxMaUserData.getTaskInvited());
        userParam.setLocationCode(wxMaUserData.getLocationCode());
        userParam.setLocationName(wxMaUserData.getLocationName());
        return userParam;
    }

    @Override
    public Boolean setUserFormId(Long userId, String formId, String appId) {
        if (StringUtils.isBlank(appId)) {
            appId = wechatProperties.getAppId();
        }
        //适配多个小程序模板消息推送
        KeyGenerator key = RedisConfig.WEIXIN_FORMID.copy().appendKey(appId).appendKey(userId);
        redisListAdapter.rightPush(key, formId);
        redisListAdapter.expire(key, 7 * 24 * 3600);

        return true;
    }

    @Override
    public String getWeChatToken() {
        try {
            return getWxMaService(null).getAccessToken();
        } catch (WxErrorException e) {
            logger.error("获取小程序的appid失败");
            return null;
        }

    }

    @Override
    public void emptyWeChatToken() {
    }

    @Override
    public Boolean userPopup(String openId, Integer platform) {
        if (redisStringAdapter.hasKey(getUserPopupKey(openId, platform))) {
            return false;
        }
        //有效期至凌晨0点
        redisStringAdapter.set(getUserPopupKey(openId, platform), 1, DateUtils.getCurSeconds());

        return true;
    }

    @Override
    public WxMaPhoneNumberInfo getUserPhoneNum(WxMaUserData wxMaUserData) {
        return getWxMaService(wxMaUserData.getAppId()).getUserService().getPhoneNoInfo(
                wxMaUserData.getSessionKey(), wxMaUserData.getEncryptedData(), wxMaUserData.getIv());
    }

    /**
     * 获取对应平台的用户弹窗key
     *
     * @param openId
     * @param platform
     * @return
     */
    private KeyGenerator getUserPopupKey(String openId, Integer platform) {
        return RedisConfig.WX_APPLET_POPUP.copy().appendKey(platform).appendKey("_").appendKey(openId);

    }

    @Override
    public Boolean hasSensitiveWord(String content) {
        String accessToken = geAppletAccessToken();
        String url = BASE_SENSITIVE_WORD_URL + accessToken;
        JSONObject para = new JSONObject();
        para.put("content", content);
        String result = httpClientService.doPostJson(url, para.toJSONString());
        WechatBackResult backResult = JSONObject.parseObject(result, WechatBackResult.class);
        return !WechatBackResult.SUCCESS_CODE.equals(backResult.getErrcode());
    }

    private String geAppletAccessToken() {
        String accessToken = null;
        try {
            accessToken = getWxMaService(wechatProperties.getWstAppId()).getAccessToken();
        } catch (WxErrorException e) {
            logger.error("获取小程序的access_token失败");
        }
        return accessToken;
    }

    @Override
    public String getOfficialAccountAppId(String type) {
        Map<String, String> appIdByType = wechatMpConfig.getAppIdByType();
        return appIdByType.get(type);
    }
}
