package com.bxm.localnews.thirdparty.service.jump;

import com.bxm.localnews.thirdparty.constant.OrderJumpType;
import com.bxm.localnews.thirdparty.param.JumpInfoConvertParam;
import com.bxm.localnews.thirdparty.service.jump.exception.ConvertNotFoundException;
import com.bxm.localnews.thirdparty.vo.JumpInfo;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Component
public class JumpInfoConvertService {

    private static final Map<OrderJumpType,JumpInfoConvert> JUMP_TYPE_JUMP_INFO_CONVERT_MAP = Maps.newHashMap();

    /**
     * 转换方法定义
     * @param param 包含用户信息、基本请求参数
     * @return 转换后的包含了跳转协议的对象
     * @throws ConvertNotFoundException 赚不到可处理的convert
     */
    public Optional<JumpInfo> convert(JumpInfoConvertParam param) {
        Optional<OrderJumpType> orderJumpType = OrderJumpType.typeValueOf(param.getJumpType());
        if (!orderJumpType.isPresent()) {
            throw new ConvertNotFoundException(String.format("类型: %s没有可用的JumpInfoConvert", param.getJumpType()));
        }

        return getConvert(orderJumpType.get()).convert(param);
    }

    private JumpInfoConvert getConvert(OrderJumpType jumpType) {

        JumpInfoConvert jumpInfoConvert = JUMP_TYPE_JUMP_INFO_CONVERT_MAP.get(jumpType);
        if (Objects.isNull(jumpInfoConvert)) {
            throw new ConvertNotFoundException(String.format("类型: %s没有可用的JumpInfoConvert", jumpType));
        }
        return jumpInfoConvert;
    }

    @EventListener(ContextRefreshedEvent.class)
    public void onContextRefreshedEvent(ContextRefreshedEvent event) {
        Collection<JumpInfoConvert> values = event.getApplicationContext().getBeansOfType(JumpInfoConvert.class).values();
        if (CollectionUtils.isEmpty(values)) {
            return;
        }

        values.forEach(p -> {
            JUMP_TYPE_JUMP_INFO_CONVERT_MAP.put(p.support(), p);
        });

    }
}
