package com.bxm.localnews.thirdparty.service.pop.popinstance.impl;

import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.dto.LocationDetailDTO;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.dto.VipInfoDetailDTO;
import com.bxm.localnews.integration.UserVipIntegrationService;
import com.bxm.localnews.param.VipQueryParam;
import com.bxm.localnews.thirdparty.param.HomeWindowParam;
import com.bxm.localnews.thirdparty.service.pop.popinstance.PopCache;
import com.bxm.localnews.thirdparty.service.pop.popinstance.PopContext;
import com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Objects;

import static com.bxm.localnews.thirdparty.service.pop.popstrategy.impl.AbstractPopStrategy.KEY;
import static com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum.VER_3_6_0;
import static com.bxm.newidea.component.tools.BitOperatorUtil.getBitAsBoolean;
import static com.bxm.newidea.component.tools.StringUtils.isGrateOrEqualThan;
import static com.bxm.newidea.component.tools.StringUtils.isLessThan;
import static java.util.Objects.isNull;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * 3.5.0引导弹窗的公共类
 * 主要是封装一些公共的流程
 *
 * @author gonzo
 * @date 2020-08-22 12:05
 **/
@Slf4j
public abstract class AbstractGuidePopup extends AbstractPop {

    /**
     * 3.5.0版本
     */
    protected static final String VER_3_5_0 = "3.5.0";

    /**
     * 3.9.0版本
     */
    protected static final String VER_3_9_0 = "3.9.0";

    @Autowired
    private LocationFacadeService locationFacadeService;

    @Autowired
    private UserVipIntegrationService userVipIntegrationService;

    static final String LOCATION_DETAIL_DTO = "LOCATION_DETAIL_DTO";

    @Override
    public boolean beforeFilter(PopContext context) {

        // 判断版本 如果版本不是要弹出的版本 则直接记录缓存，且不再弹出
        HomeWindowParam param = context.getHomeWindowParam();
        // 判断版本
        if (isBlank(param.getCurVer())
                // 如果是没配，或者小于指定的最小版本
                || (isNotBlank(afterIncludeVersion()) && isLessThan(param.getCurVer(), afterIncludeVersion())
                // 或者 没配大于或者大于等于指定的最大版本
                || (isNotBlank(beforeExcludeVersion()) && isGrateOrEqualThan(param.getCurVer(), beforeExcludeVersion())))) {

            // 如果当前弹窗不在指定版本里面，进行缓存添加

            // 用户弹窗缓存map
            String key = context.getParam(KEY);
            if (isNotBlank(key)) {
                PopCache cache = context.getCacheMap().get(key);
                // 如果弹窗缓存为空 或者没弹出来过(缓存里存着可以弹出)
                if (isNull(cache) || cache.isPopUp()) {
                    // 调用父类方法增加缓存
                    super.beforeFilter(context);
                    super.afterFilter(context);
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("请求参数: {} 关闭引导弹窗", param.getCurVer());
            }
            // 然后不用继续弹出了
            return false;
        }

        // 否则走默认的流程
        return super.beforeFilter(context);
    }

    @Override
    boolean filter(PopContext context) {
        HomeWindowParam param = context.getHomeWindowParam();
        if (isNull(param)) {
            return false;
        }

        // 版本判断 避免老版本的APP注册
        if (!checkVersion(context)) {
            log.info("用户: {} 的当前版本: {}，增加弹出缓存，永不弹出", param.getUserId(), param.getCurVer());
            closeCache(context);
            return false;
        }

        // 判断是否开通城市
        LocationDetailDTO locationDetailDTO = locationFacadeService.getLocationDetailByCode(param.getAreaCode());
        if (isNull(locationDetailDTO) || !Objects.equals(locationDetailDTO.getEnableCommunityContent(), 1)) {
            if (log.isDebugEnabled()) {
                log.debug("当前城市: {} 不是开通城市，不弹出引导弹窗", param.getAreaCode());
            }
            return false;
        }

        context.getParamMap().put(LOCATION_DETAIL_DTO, locationDetailDTO);
        return true;
    }

    /**
     * 获取当前引导弹窗在哪个版本之后弹出
     * curVersion >= afterIncludeVersion 则弹出
     *
     * @return 版本 可以设置为空
     */
    protected String afterIncludeVersion() {
        return VER_3_5_0;
    }

    /**
     * 获取当前引导弹窗在哪个版本之前弹出
     * curVersion < beforeExcludeVersion 则弹出
     *
     * @return 版本 可以设置为空
     */
    protected String beforeExcludeVersion() {
        return VER_3_9_0;
    }

    /**
     * 判断是否3.5.0以后注册的版本
     *
     * @param judgeMarket
     * @return true 是3.5.0以后新用户
     */
    protected boolean check350JudgeMarker(Long judgeMarket) {
        // 判断用户是新版本注册用户
        return getBitAsBoolean(judgeMarket, LocalNewsUserJudgeMarkerEnum.VER_3_5_0.getIndex());
    }

    /**
     * 是否3.6.0以后注册的用户
     *
     * @param judgeMarket
     * @return true 是3.6.0以后新用户
     */
    protected boolean check360JudgeMarker(Long judgeMarket) {
        // 判断用户是新版本注册用户
        return getBitAsBoolean(judgeMarket, VER_3_6_0.getIndex());
    }

    /**
     * 是否3.9.0以后注册的用户
     *
     * @param judgeMarket
     * @return true 是3.9.0以后新用户
     */
    protected boolean check390JudgeMarker(Long judgeMarket) {
        // 判断用户是新版本注册用户
        return getBitAsBoolean(judgeMarket, LocalNewsUserJudgeMarkerEnum.VER_3_9_0.getIndex());
    }


    /**
     * 判断当前版本是否符合弹出
     *
     * @param context context
     * @return true 是
     */
    protected boolean checkVersion(PopContext context) {
        String curVersion = context.getHomeWindowParam().getCurVer();
        return isNotBlank(curVersion)
                // curVersion >= afterIncludeVersion
                && (isBlank(afterIncludeVersion()) || isGrateOrEqualThan(curVersion, afterIncludeVersion()))
                // curVersion < beforeExcludeVersion
                && (isBlank(beforeExcludeVersion()) || isLessThan(curVersion, beforeExcludeVersion()));
    }


    /**
     * 是否是激活VIP
     *
     * @param userInfo
     * @return
     */
    protected boolean isInviteVip(UserInfoDTO userInfo) {
        if (isNull(userInfo)) {
            return false;
        }

        // 非VIP
        if (!Objects.equals(userInfo.getIsVip(), 1)) {
            return false;
        }

        // 查询VIP服务，是否是激活（激活码激活）的VIP
        VipQueryParam param = new VipQueryParam();
        param.setUserId(userInfo.getId());
        VipInfoDetailDTO vipDetail = userVipIntegrationService.getVipDetail(param);

        // 0: 自费  非自费则就是邀请激活VIP
        return !Objects.equals(vipDetail.getType(), 0);
    }

    @Override
    public void afterFilter(PopContext context) {
        // do noting 这个弹窗是需要手动上报后关闭的
        if (autoCache()) {
            super.afterFilter(context);
        }
    }

    /**
     * 是否自动添加弹窗缓存
     * 默认false
     *
     * @return true 是
     */
    protected boolean autoCache() {
        return false;
    }

    /**
     * 添加弹出缓存
     *
     * @param context context
     */
    protected void closeCache(PopContext context) {
        super.afterFilter(context);
    }
}
