package com.bxm.localnews.user.service.handler;

import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.param.NativeRecommendContext;
import com.bxm.localnews.user.properties.RecommendProperties;
import com.bxm.localnews.user.service.intefaces.BarrelHandler;
import com.bxm.localnews.user.service.intefaces.WeightSupport;
import com.bxm.localnews.user.vo.RecommendNative;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.BaseParam;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import java.util.List;

@Slf4j
public abstract class AbstractWeightBarrelHandler<T extends BaseParam> implements BarrelHandler<BaseParam>, WeightSupport {

    @Autowired
    protected RecommendProperties recommendProperties;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private RedisSetAdapter redisSetAdapter;

    @Override
    public <R> R handle(BaseParam baseParam) {
        T param = (T) baseParam;

        //获得推荐数据
        R result = doHandle(param);

        //后置操作
        processAfter(param);

        return result;
    }

    protected abstract <R> R doHandle(T t);


    private void processAfter(T t) {
        NativeRecommendContext param = (NativeRecommendContext) t;

        List<RecommendNative> recommendNativeList = param.getResult();

        KeyGenerator recommendKey = RedisConfig.USER_RECOMMEND.copy()
                .appendKey("recommended")
                .appendKey(param.getRecommendCategory())
                .appendKey(param.getUserId());
        //将结果存至redis中
        if (!CollectionUtils.isEmpty(recommendNativeList)) {
            redisSetAdapter.add(recommendKey, recommendNativeList.parallelStream().toArray(Object[]::new));
            redisSetAdapter.expire(recommendKey, recommendProperties.getCacheInterval());
        } else {
            redisSetAdapter.remove(recommendKey);
        }

    }

    @Override
    public Boolean support(BaseParam baseParam) {
        NativeRecommendContext param = (NativeRecommendContext) baseParam;
        Long userId = param.getUserId();
        String areaCode = param.getCurrentAreaCode();
        //获取分桶策略js脚本,支持在线切换
        return getResultFromJs(areaCode, userId);
    }

    /**
     * 分桶策略，从redis中根据不同的桶获取不同桶的计算方式，从而使得用户根据id以及地区编码
     * 能够到达不同的桶策略中。
     *
     * @param areaCode 当前地区的编码
     * @param userId   用户id
     * @return 是否满足
     */
    @SneakyThrows
    private Boolean getResultFromJs(String areaCode, Long userId) {
        //得到js代码（such as -> function judge(userId,areaCode){ return (userId & 1)==1); }）接受奇数的用户id
        String formula = recommendProperties.getFormula().getOrDefault(supportWeight(), "").toString();
        if (StringUtils.isBlank(formula)) {
            return true;
        }
        // 创建脚本引擎管理器
        ScriptEngineManager sem = new ScriptEngineManager();
        // 创建一个处理JavaScript的脚本引擎
        ScriptEngine engine = sem.getEngineByExtension("js");
        // 执行js公式
        engine.put("userId", userId);
        engine.put("areaCode", areaCode);
        Object result = null;
        try {
            result = engine.eval(formula);
        } catch (ScriptException e) {
            log.error("js脚本执行错误,内容为:[{}]", formula);
        }
        return (Boolean) result;
    }

    @Override
    public int getOrder() {
        return HandlerOrder.get(this.getClass());
    }

}
