package com.bxm.openlog.sdk.scheduler;

import com.bxm.openlog.sdk.listener.eventbus.EventBusSubscriberMessageListener;
import com.bxm.openlog.sdk.listener.eventbus.OpenLogEventBusFactory;
import com.bxm.warcar.integration.eventbus.AsyncEventPark;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.mq.SingleMessageListener;
import com.bxm.warcar.mq.rocketmq.RocketmqConsumer;
import com.bxm.warcar.utils.NamedThreadFactory;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationListener;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author zhangdong
 * @date 2023/12/4
 */
@Slf4j
@ConditionalOnClass({RocketmqConsumer.class,OpenLogEventBusFactory.class})
@ConditionalOnProperty(value = MqFlowControlProperties.PROPERTY_ENABLE, havingValue = "true")
@EnableConfigurationProperties(MqFlowControlProperties.class)
public class MqFlowControlScheduler implements ApplicationListener<ApplicationReadyEvent>{
    private final Map<DefaultMQPushConsumer, AsyncEventPark> map = new HashMap<>();
    private final ScheduledExecutorService scheduledExecutorService =
            new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("MqFlowControlScheduler"));
    private final MqFlowControlProperties properties;

    public MqFlowControlScheduler(MqFlowControlProperties properties) {
        this.properties = properties;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        ConfigurableListableBeanFactory beanFactory = event.getApplicationContext().getBeanFactory();
        List<RocketmqConsumer> rocketmqConsumers = beanFactory.getBeansOfType(OpenLogEventBusFactory.class)
                .values().stream().filter(x -> x.getConsumerMap() != null).flatMap(x -> x.getConsumerMap().values().stream())
                .collect(Collectors.toList()).stream().filter(x -> x instanceof RocketmqConsumer).map(x -> (RocketmqConsumer) x).collect(Collectors.toList());
        for (RocketmqConsumer rocketmqConsumer : rocketmqConsumers) {
            SingleMessageListener messageListener = rocketmqConsumer.getMessageListener();
            if (!(messageListener instanceof EventBusSubscriberMessageListener)) {
                continue;
            }
            EventBusSubscriberMessageListener listener = (EventBusSubscriberMessageListener) messageListener;
            EventPark eventPark = listener.getEventPark();
            if (!(eventPark instanceof AsyncEventPark)) {
                continue;
            }
            DefaultMQPushConsumer consumer = rocketmqConsumer.getConsumer();
            map.put(consumer, (AsyncEventPark) eventPark);
        }
        if (map.size() > 0) {
            startScheduler();
        }
    }

    private void startScheduler() {
        scheduledExecutorService.scheduleAtFixedRate(() -> {
            for (Map.Entry<DefaultMQPushConsumer, AsyncEventPark> entry : map.entrySet()) {
                DefaultMQPushConsumer consumer = entry.getKey();
                boolean pause = consumer.getDefaultMQPushConsumerImpl().isPause();
                int queueSize = entry.getValue().getQueueSize();
                int threshold = properties.getThreshold();
                if (queueSize >= threshold && !pause) {
                    consumer.suspend();
                    log.info("Consumer suspend, consumerGroup: {}, queueSize: {}", consumer.getConsumerGroup(), queueSize);
                }
                if (queueSize < threshold && pause) {
                    consumer.resume();
                    log.info("Consumer resume, consumerGroup: {}, queueSize: {}", consumer.getConsumerGroup(), queueSize);
                }
            }
        }, 0, properties.getIntervalTime(), TimeUnit.SECONDS);
    }
}
