package com.bxm.pangu.rta.common.taobao;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.pangu.rta.common.*;
import com.bxm.warcar.utils.SafeMapHelper;
import com.google.common.collect.Maps;
import com.taobao.api.ApiException;
import com.taobao.api.DefaultTaobaoClient;
import com.taobao.api.TaobaoClient;
import com.taobao.api.internal.util.TaobaoLogger;
import com.taobao.api.request.UsergrowthDhhDeliveryAskRequest;
import com.taobao.api.response.UsergrowthDhhDeliveryAskResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 淘宝官方 - 大航海
 * @author allen
 * @date 2021-10-13
 * @since 1.0
 */
@Slf4j
public class TaobaoRtaClient implements RtaClient {

    private final TaobaoRtaProperties properties;
    private final ConcurrentHashMap<String, TaobaoClient> clients = new ConcurrentHashMap<>();

    public TaobaoRtaClient(TaobaoRtaProperties properties) {
        this.properties = properties;
        TaobaoLogger.setNeedEnableLogger(properties.isNeedEnableLogger());
    }

    @Override
    public String getIdentifyForCounter(RtaRequest request) {
        String param = request.getParam();
        String appKey = properties.getAppKey();
        if (StringUtils.isNotBlank(param)) {
            String[] params = StringUtils.splitPreserveAllTokens(param, "|");
            if (params.length > 4) {
                appKey = params[4];
            }
        }
        return appKey;
    }

    private TaobaoClient getClient(String appKey, String appSecret) {
        return SafeMapHelper.get(clients, appKey, new SafeMapHelper.InitializingValue<TaobaoClient>() {
            @Override
            public TaobaoClient initializing() {
                String url = properties.getUrl();
                int connectTimeout = properties.getConnectTimeout();
                int socketTimeout = properties.getSocketTimeout();
                return new DefaultTaobaoClient(url, appKey, appSecret, "json", connectTimeout, socketTimeout);
            }
        });
    }

    @Override
    public boolean isTarget(RtaRequest req, Consumer<Map<Object, Object>> feedback) throws RtaRequestException {
        UsergrowthDhhDeliveryAskRequest request = new UsergrowthDhhDeliveryAskRequest();
        if (StringUtils.equals(req.getOs(), RtaRequest.Os.ANDROID)) {
            request.setOs("0");
        } else if (StringUtils.equals(req.getOs(), RtaRequest.Os.IOS)) {
            request.setOs("1");
        } else {
            request.setOs("3");
        }
        String imei = req.getImei();
        if (StringUtils.isNotBlank(imei)) {
            request.setImei(imei);
        }
        String imeiMd5 = req.getImei_md5();
        if (StringUtils.isNotBlank(imeiMd5)) {
            request.setImeiMd5(imeiMd5);
        }
        String idfa = req.getIdfa();
        if (StringUtils.isNotBlank(idfa)) {
            request.setIdfa(idfa);
        }
        String idfaMd5 = req.getIdfa_md5();
        if (StringUtils.isNotBlank(idfaMd5)) {
            request.setIdfaMd5(idfaMd5);
        }
        String oaid = req.getOaid();
        if (StringUtils.isNotBlank(oaid)) {
            request.setOaid(oaid);
        }
        String oaidMd5 = req.getOaid_md5();
        if (StringUtils.isNotBlank(oaidMd5)) {
            request.setOaidMd5(oaidMd5);
        }
        String channel = properties.getChannel();
        String advertisingSpaceId = properties.getAdvertisingSpaceId();
        String targetTaskId = null;
        String appKey = properties.getAppKey();
        String appSecret = properties.getAppSecret();
        String param = req.getParam();
        if (StringUtils.isNotBlank(param)) {
            // param:
            // e.g. channel
            // e.g. channel|advertisingSpaceId|targetTaskId
            // e.g. |advertisingSpaceId|targetTaskId
            String[] params = StringUtils.splitPreserveAllTokens(param, "|");
            channel = StringUtils.defaultIfBlank(params[0], properties.getChannel());
            if (params.length > 1) {
                advertisingSpaceId = StringUtils.defaultIfBlank(params[1], properties.getAdvertisingSpaceId());
            }
            if (params.length > 2) {
                targetTaskId = params[2];
            }
            if (params.length > 3) {
                appKey = params[3];
            }
            if (params.length > 4) {
                appSecret = params[4];
            }
        }
        request.setChannel(channel);
        request.setAdvertisingSpaceId(advertisingSpaceId);

        try {
            TaobaoClient client = getClient(appKey, appSecret);
            if (Objects.isNull(client)) {
                throw new NullPointerException(String.format("Not found client of %s %s", appKey, appSecret));
            }
            UsergrowthDhhDeliveryAskResponse response = client.execute(request);
            if (Objects.isNull(response)) {
                return false;
            }
            String body = response.getBody();
            if (log.isDebugEnabled()) {
                log.debug("response: {}", body);
            }
            boolean hitTarget = Optional.ofNullable(response.getResult()).orElse(false);
            if (!hitTarget) {
                return false;
            }
            // {"usergrowth_dhh_delivery_ask_response":{"errcode":0,"result":true,"task_id_list":{"string":["1163346009","1763656536","1981891941"]},"request_id":"66hyfflhb6gi"}}
            JSONObject json = JSONObject.parseObject(body);
            JSONObject usergrowthDhhDeliveryAskResponse = json.getJSONObject("usergrowth_dhh_delivery_ask_response");
            if (MapUtils.isEmpty(usergrowthDhhDeliveryAskResponse)) {
                return false;
            }
            JSONObject taskIdListNode = usergrowthDhhDeliveryAskResponse.getJSONObject("task_id_list");
            if (MapUtils.isEmpty(taskIdListNode)) {
                return false;
            }
            JSONArray taskIdListJsonArray = taskIdListNode.getJSONArray("string");
            Set<String> taskIdList = taskIdListJsonArray.stream().map(new Function<Object, String>() {
                @Override
                public String apply(Object object) {
                    return Objects.toString(object);
                }
            }).collect(Collectors.toSet());

            String taskId = response.getTaskId();
            // 如果需要检查目标，但是不在 taskIdList，则表示未命中
            if (StringUtils.isNotBlank(targetTaskId) && !taskIdList.contains(targetTaskId)) {
                return false;
            }

            Map<Object, Object> res = Maps.newHashMap();
            res.put("channel", channel);
            res.put("advertisingSpaceId", advertisingSpaceId);
            res.put("taskId", taskId);
            res.put("taskIdList", taskIdList);
            res.put("device", req);

            Function<Map<Object, Object>, Boolean> targetHandler = properties.getTargetHandler();
            boolean isTarget = Objects.nonNull(targetHandler) ? targetHandler.apply(res) : true;

            // 否则不检查目标，那到这里表示命中了。
            if (isTarget && Objects.nonNull(feedback)) {
                feedback.accept(res);
            }
            return isTarget;
        } catch (ApiException e) {
            log.warn("{}", e.getMessage());
            throw new RtaRequestException(e);
        }
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Taobao;
    }

    @Override
    public RtaClientProperties getProperties() {
        return properties;
    }
}
