package com.bxm.pangu.rta.common.tianyancha;

import com.bxm.pangu.rta.common.AbstractHttpClientRtaClient;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.pangu.rta.common.RtaRequestException;
import com.bxm.pangu.rta.common.RtaType;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.StringHelper;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.experimental.Accessors;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.HmacAlgorithms;
import org.apache.commons.codec.digest.HmacUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;

import java.util.List;

/**
 * 天眼查
 *
 * @author allen
 * @date 2022-06-30
 * @since 1.0
 */
public class TianYanChaRtaClient extends AbstractHttpClientRtaClient {

    private static final String IOS = "ios";
    private static final String ANDROID = "android";
    private static final String OTHERS = "others";

    public TianYanChaRtaClient(TianYanChaRtaProperties properties) {
        super(properties);
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        String param = request.getParam();

        String[] arr = StringUtils.split(param, "|");
        if (ArrayUtils.getLength(arr) < 3) {
            throw new RtaRequestException("param");
        }
        String reqId = arr[0];
        String campaignId = arr[1];
        String secretKey = arr[2];

        Request req = new Request();
        req.setReq_id(reqId);
        req.setReq_time(System.currentTimeMillis() / 1000);
        req.setCampaign_id(campaignId);

        List<User> users = Lists.newArrayList();
        // imei 原文，md5 都支持
        // oaid / idfa 只支持 md5的

        // 系统类型：ios/android/others
        if (StringUtils.isNotBlank(request.getImei())) {
            users.add(new User().setOs_type(ANDROID).setDevice_id(request.getImei()));
        }
        if (StringUtils.isNotBlank(request.getImei_md5())) {
            users.add(new User().setOs_type(ANDROID).setDevice_id(request.getImei_md5()));
        }
        if (StringUtils.isNotBlank(request.getOaid_md5())) {
            users.add(new User().setOs_type(OTHERS).setDevice_id(request.getOaid_md5()));
        }
        if (StringUtils.isNotBlank(request.getIdfa_md5())) {
            users.add(new User().setOs_type(IOS).setDevice_id(request.getIdfa_md5()));
        }

        req.setUsers(users);

        String unstr = req.getReq_id() + req.getReq_time() + req.getCampaign_id();
        byte[] unbyte = StringHelper.convert(unstr);
        byte[] hmac = new HmacUtils(HmacAlgorithms.HMAC_SHA_256, StringHelper.convert(secretKey)).hmac(unbyte);
        String signature = Base64.encodeBase64String(hmac);

        req.setSignature(signature);

        HttpPost post = new HttpPost(getProperties().getUrl());
        post.setEntity(new ByteArrayEntity(JsonHelper.convert2bytes(req)));
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
        return post;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        return JsonHelper.convert(json, Response.class).isTarget();
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.TianYanCha;
    }

    @Data
    private static class Request {
        private String req_id;
        private List<User> users;
        private long req_time;
        private String campaign_id;
        private String signature;
    }

    @Data
    @Accessors(chain = true)
    private static class User {
        private String os_type;
        private String device_id;
    }

    @Data
    private static class Response {
        private long rsp_time;
        private String result_code;
        private String result_message;
        private List<Material> materials;

        private boolean isTarget() {
            return StringUtils.equalsIgnoreCase("0", getResult_code()) && CollectionUtils.isNotEmpty(getMaterials());
        }
    }

    @Data
    private static class Material {
        private String device_id;
    }
}
