package com.bxm.pangu.rta.common.taobao_vegas;

import com.bxm.pangu.rta.common.*;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.SafeMapHelper;
import com.google.common.collect.Maps;
import com.taobao.api.DefaultTaobaoClient;
import com.taobao.api.TaobaoClient;
import com.taobao.api.internal.util.TaobaoLogger;
import com.taobao.api.request.TbkDgVegasSendStatusRequest;
import com.taobao.api.response.TbkDgVegasSendStatusResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

/**
 * @author allen
 * @date 2022-04-11
 * @since 1.0
 */
@Slf4j
public class TaobaoVegasRtaClient implements RtaClient {

    private final TaobaoVegasRtaProperties properties;
    private final ConcurrentHashMap<String, TaobaoClient> clients = new ConcurrentHashMap<>();

    public TaobaoVegasRtaClient(TaobaoVegasRtaProperties properties) {
        this.properties = properties;
        TaobaoLogger.setNeedEnableLogger(properties.isNeedEnableLogger());
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.TaobaoVegas;
    }

    @Override
    public RtaClientProperties getProperties() {
        return properties;
    }

    private TaobaoClient getClient(String appKey, String appSecret) {
        return SafeMapHelper.get(clients, appKey, (SafeMapHelper.InitializingValue<TaobaoClient>) () -> {
            String url = properties.getUrl();
            int connectTimeout = properties.getConnectTimeout();
            int socketTimeout = properties.getSocketTimeout();
            return new DefaultTaobaoClient(url, appKey, appSecret, "json", connectTimeout, socketTimeout);
        });
    }

    @Override
    public boolean isTarget(RtaRequest request, Consumer<Map<Object, Object>> feedback) throws RtaRequestException {
        String param = request.getParam();
        String[] arr = StringUtils.splitPreserveAllTokens(param, "|");

        // {relation_id}|{special_id}|{thor_biz_code}|{pid}|{appKey}|{appSecret}
        if (ArrayUtils.isEmpty(arr) || arr.length < 6) {
            throw new RtaRequestException("param");
        }
        String relationId = arr[0];
        String specialId = arr[1];
        String thorBizCode = arr[2];
        String pid = arr[3];
        String appKey = arr[4];
        String appSecret = arr[5];

        TbkDgVegasSendStatusRequest req = new TbkDgVegasSendStatusRequest();
        if (StringUtils.isNotBlank(relationId)) {
            req.setRelationId(relationId);
        }
        if (StringUtils.isNotBlank(specialId)) {
            req.setSpecialId(specialId);
        }
        if (StringUtils.isNotBlank(thorBizCode)) {
            req.setThorBizCode(thorBizCode);
        }
        if (StringUtils.isNotBlank(pid)) {
            req.setPid(pid);
        }

        String oaidMd5 = request.getOaid_md5();
        if (StringUtils.isBlank(req.getDeviceValue()) && StringUtils.isNotBlank(oaidMd5)) {
            req.setDeviceValue(oaidMd5);
            req.setDeviceType("OAID");
        }

        String imeiMd5 = request.getImei_md5();
        if (StringUtils.isBlank(req.getDeviceValue()) && StringUtils.isNotBlank(imeiMd5)) {
            req.setDeviceValue(imeiMd5);
            req.setDeviceType("IMEI");
        }

        String idfaMd5 = request.getIdfa_md5();
        if (StringUtils.isBlank(req.getDeviceValue()) && StringUtils.isNotBlank(idfaMd5)) {
            req.setDeviceValue(idfaMd5);
            req.setDeviceType("IDFA");
        }

        try {
            TaobaoClient client = getClient(appKey, appSecret);
            TbkDgVegasSendStatusResponse response = client.execute(req);
            TbkDgVegasSendStatusResponse.Data data = response.getData();

            if (log.isDebugEnabled()) {
                log.debug("response: {}", JsonHelper.convert(response));
            }

            boolean isTarget = Optional.ofNullable(data).map(data1 -> {
                List<TbkDgVegasSendStatusResponse.MapData> list = Optional.ofNullable(data1.getResultList()).orElse(new ArrayList<>());
                return list.stream().anyMatch(mapData -> StringUtils.equals("1", mapData.getIsNewUser()));
            }).orElse(false);
            if (isTarget && Objects.nonNull(feedback)) {
                feedback.accept(Maps.newHashMap());
            }
            return isTarget;
        } catch (Exception e) {
            throw new RtaRequestException(e);
        }
    }
}
