package com.bxm.pangu.rta.common.micrometer;

import com.bxm.pangu.rta.common.RtaClient;
import com.bxm.pangu.rta.common.RtaType;
import com.google.common.collect.Maps;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.binder.MeterBinder;
import lombok.extern.slf4j.Slf4j;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * Rta 客户端的Meter
 *
 * @author allen
 * @date 2021-08-17
 * @since 1.0
 */
@Slf4j
public class RtaClientMicroMeter implements MeterBinder {

    private Map<RtaType, Timer> timer = Maps.newHashMap();
    private Map<RtaType, Counter> success = Maps.newHashMap();
    private Map<RtaType, Counter> fail = Maps.newHashMap();
    private Map<RtaType, Counter> connectException = Maps.newHashMap();
    private Map<RtaType, Counter> readException = Maps.newHashMap();
    private Map<RtaType, Counter> otherException = Maps.newHashMap();
    /**
     * 包含缓存的计数器
     * @see com.bxm.pangu.rta.common.CachingAspect
     */
    private Map<RtaType, Counter> withCaching = Maps.newHashMap();

    private final Iterable<RtaClient> clients;

    public RtaClientMicroMeter(Iterable<RtaClient> clients) {
        this.clients = clients;
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        for (RtaClient client : clients) {
            RtaType rtaType = client.getRtaType();
            Counter successCounter = Counter.builder("rta.success").tag("name", rtaType.name()).register(registry);
            Counter failCounter = Counter.builder("rta.fail").tag("name", rtaType.name()).register(registry);
            Counter requestExceptionCounter = Counter.builder("rta.connect.ex").tag("name", rtaType.name()).register(registry);
            Counter socketTimeoutExceptionCounter = Counter.builder("rta.read.ex").tag("name", rtaType.name()).register(registry);
            Counter otherExceptionCounter = Counter.builder("rta.other.ex").tag("name", rtaType.name()).register(registry);
            Counter withCachingCounter = Counter.builder("rta.other.with.caching").tag("name", rtaType.name()).register(registry);
            Timer.Builder timerBuilder = Timer.builder("rta.requests").tag("name", rtaType.name());
            if (client.getProperties().isPublishPercentiles()) {
                timerBuilder.publishPercentiles(0.5, 0.75, 0.9, 0.95, 0.99);
            }
            Timer theTimer = timerBuilder.register(registry);

            success.put(rtaType, successCounter);
            fail.put(rtaType, failCounter);
            timer.put(rtaType, theTimer);
            connectException.put(rtaType, requestExceptionCounter);
            readException.put(rtaType, socketTimeoutExceptionCounter);
            otherException.put(rtaType, otherExceptionCounter);
            withCaching.put(rtaType, withCachingCounter);
        }
    }

    void incrementSuccess(RtaType rtaType) {
        Counter counter = success.get(rtaType);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    void incrementFail(RtaType rtaType) {
        Counter counter = fail.get(rtaType);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    void incrementConnectException(RtaType rtaType) {
        Counter counter = connectException.get(rtaType);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    void incrementReadException(RtaType rtaType) {
        Counter counter = readException.get(rtaType);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    void incrementOtherException(RtaType rtaType) {
        Counter counter = otherException.get(rtaType);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    public void incrementWithCaching(RtaType rtaType) {
        Counter counter = withCaching.get(rtaType);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    void record(RtaType rtaType, long startNanoTime) {
        Timer timer = this.timer.get(rtaType);
        if (Objects.nonNull(timer)) {
            timer.record((System.nanoTime() - startNanoTime), TimeUnit.NANOSECONDS);
        }
    }
}
