package com.bxm.pangu.rta.common.ucgf;

import com.bxm.pangu.rta.common.AbstractHttpClientRtaClient;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.pangu.rta.common.RtaRequestException;
import com.bxm.pangu.rta.common.RtaType;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.springframework.http.MediaType;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author jingyu.li
 * @date 2023-12-20
 */
@Slf4j
public class UcgfRtaClient extends AbstractHttpClientRtaClient {

    private final UcgfRtaProperties properties;

    public UcgfRtaClient(UcgfRtaProperties properties) {
        super(properties);
        this.properties = properties;
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        String param = request.getParam();
        if (StringUtils.isBlank(param)) {
            throw new RtaRequestException("param");
        }

        UcgfRtaRequest ucgfRtaRequest = new UcgfRtaRequest();
        ucgfRtaRequest.setChannel(properties.getChannel());
        switch (request.getOs()) {
            case RtaRequest.Os.ANDROID:
                ucgfRtaRequest.setPlatform("ANDROID");
                break;
            case RtaRequest.Os.IOS:
                ucgfRtaRequest.setPlatform("IOS");
                break;
            default:
                ucgfRtaRequest.setPlatform("UNKNOWN");
                break;
        }

        String did = null;
        String didType = null;

        String imei = request.getImei();
        if (StringUtils.isNotBlank(imei)) {
            did = imei;
            didType = "IMEI";
        } else if (StringUtils.isNotBlank(request.getImei_md5())) {
            did = request.getImei_md5();
            didType = "IMEI_MD5";
        } else if (StringUtils.isNotBlank(request.getOaid())) {
            did = request.getOaid();
            didType = "OAID";
        } else if (StringUtils.isNotBlank(request.getOaid_md5())) {
            did = request.getOaid_md5();
            didType = "OAID_MD5";
        } else if (StringUtils.isNotBlank(request.getAndroidid())) {
            did = request.getAndroidid();
            didType = "ANDROID_ID";
        } else if (StringUtils.isNotBlank(request.getAndroidid_md5())) {
            did = request.getAndroidid_md5();
            didType = "ANDROID_ID_MD5";
        } else if (StringUtils.isNotBlank(request.getIdfa())) {
            did = request.getIdfa();
            didType = "IDFA";
        } else if (StringUtils.isNotBlank(request.getIdfa_md5())) {
            did = request.getIdfa_md5();
            didType = "IDFA_MD5";
        }

        if (StringUtils.isBlank(did)) {
            throw new RtaRequestException("did");
        }
        ucgfRtaRequest.setDid(did);
        ucgfRtaRequest.setDidType(didType);

        String url = properties.getUrl().replaceAll("<channel>",properties.getChannel());

        HttpPost httpPost = new HttpPost(url);
        httpPost.addHeader("Content-Type",MediaType.APPLICATION_JSON_VALUE);
        httpPost.setEntity(new StringEntity(JsonHelper.convert(ucgfRtaRequest), StandardCharsets.UTF_8));
        return httpPost;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        UcgfRtaResponse convert = JsonHelper.convert(json, UcgfRtaResponse.class);
        Integer statusCode = convert.getStatus_code();
        List<Long> ucAcIds = convert.getAcIds();

        if (statusCode != null && statusCode == 0 && CollectionUtils.isNotEmpty(ucAcIds)) {
            String param = request.getParam();
            List<Long> bxmAcIds = Arrays.stream(param.split("\\|")).map(Long::valueOf).collect(Collectors.toList());

            ucAcIds.retainAll(bxmAcIds);
            return !ucAcIds.isEmpty();
        }

        return false;
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Ucgf;
    }
}
