package com.bxm.pangu.rta.common.lazada;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import com.bxm.pangu.rta.common.*;
import com.bxm.pangu.rta.common.utils.HttpClientUtils;
import com.google.common.collect.Maps;
import com.taobao.api.Constants;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.util.EntityUtils;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.*;
import java.util.function.Consumer;

/**
 * @author jingyu.li
 * @date 2023-01-05
 */
@Slf4j
public class LazadaRtaClient implements RtaClient {

    private final HttpClient httpClient;
    private final LazadaRtaProperties properties;

    public LazadaRtaClient(LazadaRtaProperties properties) {
        this.properties = properties;
        this.httpClient = HttpClientUtils.createHttpClient(properties.getMaxTotal(), properties.getDefaultMaxPerRoute(),
                properties.getConnectionRequestTimeout(), properties.getConnectTimeout(), properties.getSocketTimeout());
    }

    /**
     * 盘古调这个方法
     * @param request 请求对象
     * @param feedback 命中后的回调
     * @return
     * @throws RtaRequestException
     */
    @Override
    public boolean isTarget(RtaRequest request, Consumer<Map<Object, Object>> feedback) throws RtaRequestException {
        String body = null;
        HttpRequestBase requestBase = null;
        RtaType rtaType = getRtaType();
        try {
            requestBase = create(request);
            if (log.isDebugEnabled()) {
                log.debug("request: {}", requestBase.getURI());
            }

            HttpResponse response = httpClient.execute(requestBase);
            int statusCode = response.getStatusLine().getStatusCode();

            if (statusCode != HttpStatus.SC_OK) {
                log.warn("{} - statusCode: {}", rtaType, statusCode);
            }

            body = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
            if (log.isDebugEnabled()) {
                log.debug("response: {}", body);
            }
            boolean isTarget = isTarget(request, body,feedback);
            return isTarget;
        } catch (JSONException e) {
            if (properties.isPrintException()) {
                log.warn("{} - {} - {}", rtaType, e.getMessage(), body);
            }
            throw new RtaRequestException(e);
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("{} - {}", rtaType, e.getMessage());
            }
            throw new RtaRequestException(e);
        } finally {
            if (Objects.nonNull(requestBase) && !requestBase.isAborted()) {
                requestBase.abort();
            }
        }
    }

    /**
     * 创建请求
     * @param request
     * @return
     */
    private HttpRequestBase create(RtaRequest request) {
        String param = request.getParam();
        if (StringUtils.isBlank(param)) {
            throw new RtaRequestException("param cannot be empty");
        }
        MultiValueMap<String, String> params = new LinkedMultiValueMap<>();
        params.add("app_key",properties.getAppKey());
        params.add("timestamp",String.valueOf(System.currentTimeMillis()));
//        params.add("sign_method","HmacSHA256");
        params.add("sign_method",Constants.SIGN_METHOD_HMAC_SHA256);
        if (StringUtils.isNotBlank(request.getOaid())) {
            params.add("oaid",request.getOaid());
        }
        if (StringUtils.isNotBlank(request.getOaid_md5())) {
            params.add("oaid_md5",request.getOaid_md5());
        }
        if (StringUtils.isNotBlank(request.getGaid())) {
            params.add("gaid",request.getGaid());
        }
        if (StringUtils.isNotBlank(request.getGaid_md5())) {
            params.add("gaid_md5",request.getGaid_md5());
        }
        if (StringUtils.isNotBlank(request.getIdfa())) {
            params.add("idfa",request.getIdfa());
        }
        if (StringUtils.isNotBlank(request.getIdfa_md5())) {
            params.add("idfa_md5",request.getIdfa_md5());
        }
        if (StringUtils.isNotBlank(request.getImei())) {
            params.add("imei",request.getImei());
        }
        JSONObject jsonObject = JSONObject.parseObject(properties.getUrlJson());
        String campaignIdList = JSONObject.toJSONString(jsonObject.keySet());
        params.add("campaign_id_list",campaignIdList);
        params.add("country",request.getCountryCode());
        String memberId = JSONObject.parseObject(param).getString(request.getCountryCode());
        params.add("member_id",memberId);
        String sign = null;
        try {
            sign = signApiRequest(params, null, properties.getAppSecret(), properties.getApiName());
        } catch (IOException e) {
            throw new RtaRequestException("sign error " + e.getMessage());
        }
        params.remove("campaign_id_list");
        params.add("campaign_id_list",URLEncoder.encode(campaignIdList));
        params.add("sign",sign);

        String url = properties.getUrl() + properties.getApiName();

        UriComponentsBuilder uriComponentsBuilder = UriComponentsBuilder.fromUriString(url).replaceQueryParams(params);

        return new HttpGet(uriComponentsBuilder.build().toString());
    }

    /**
     * 判断返回值
     * @param request
     * @param json
     * @param feedback
     * @return
     */
    private boolean isTarget(RtaRequest request, String json,Consumer<Map<Object, Object>> feedback) {
        if (StringUtils.isBlank(json)) {
            return false;
        }
        JSONObject response = JSONObject.parseObject(json);
        if (!StringUtils.equals("0",response.getString("code"))) {
            return false;
        }
        JSONObject data = response.getJSONObject("data");
        JSONArray targetList = data.getJSONArray("target_list");
        if (targetList.isEmpty()) {
            return false;
        }
        /**
         * 配置文件配置的campaignId-》deeplink、clickUrl
         */
        JSONObject urlJson = JSONObject.parseObject(properties.getUrlJson());
        for (int i = 0; i < targetList.size(); i++) {
            JSONObject jsonObject = targetList.getJSONObject(i);
            if (!jsonObject.getBoolean("target")) {
                continue;
            }
            String campaignId = jsonObject.getString("campaign_id");
            JSONArray urlJsonArray = urlJson.getJSONArray(campaignId);
            if (Objects.nonNull(feedback)) {
                HashMap<Object, Object> res = Maps.newHashMap();
                res.put("token",data.getString("token"));
                res.put("deepLinkUrl",urlJsonArray.getString(0));
                res.put("clickUrl",urlJsonArray.getString(1));
                feedback.accept(res);
                return true;
            }
        }
        return false;
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Lazada;
    }

    @Override
    public RtaClientProperties getProperties() {
        return properties;
    }

    /**
     * Sign the API request with body.
     */
    public static String signApiRequest(MultiValueMap<String, String> params, String body, String appSecret, String apiName) throws IOException {
        // first: sort all text parameters
        String[] keys = params.keySet().toArray(new String[0]);
        Arrays.sort(keys);

        // second: connect all text parameters with key and value
        StringBuilder query = new StringBuilder();
        query.append(apiName);
        for (String key : keys) {
            String value = params.getFirst(key);
            if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)) {
                query.append(key).append(value);
            }
        }

        // third：put the body to the end
        if (body != null) {
            query.append(body);
        }

        // next : sign the whole request
        byte[] bytes = encryptHMACSHA256(query.toString(), appSecret);

        // finally : transfer sign result from binary to upper hex string
        return byte2hex(bytes);
    }


    private static byte[] encryptHMACSHA256(String data, String secret) throws IOException  {
        byte[] bytes = null;
        try {
            SecretKey secretKey = new SecretKeySpec(secret.getBytes(Constants.CHARSET_UTF8), "HmacSHA256");
            Mac mac = Mac.getInstance(secretKey.getAlgorithm());
            mac.init(secretKey);
            bytes = mac.doFinal(data.getBytes(Constants.CHARSET_UTF8));
        } catch (GeneralSecurityException gse) {
            throw new IOException(gse.toString());
        }
        return bytes;
    }

    /**
     * Transfer binary array to HEX string.
     */
    public static String byte2hex(byte[] bytes) {
        StringBuilder sign = new StringBuilder();
        for (int i = 0; i < bytes.length; i++) {
            String hex = Integer.toHexString(bytes[i] & 0xFF);
            if (hex.length() == 1) {
                sign.append("0");
            }
            sign.append(hex.toUpperCase());
        }
        return sign.toString();
    }
}
