package com.bxm.pangu.rta.common.qihang;

import com.bxm.pangu.rta.common.*;
import com.bxm.pangu.rta.common.qihang.proto.SurgeBidding;
import com.bxm.pangu.rta.common.qihang.request.BidRequest;
import com.bxm.pangu.rta.common.utils.HttpClientUtils;
import com.bxm.warcar.utils.JsonHelper;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.util.EntityUtils;
import org.jsoup.helper.StringUtil;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * @author jingyu.li
 * @date 2024-03-11
 */
@Slf4j
public class QiHangRtaClient implements RtaClient {

    private final HttpClient httpClient;
    private final QiHangRtaProperties properties;

    public QiHangRtaClient(QiHangRtaProperties properties) {
        this.properties = properties;
        this.httpClient = HttpClientUtils.createHttpClient(properties.getMaxTotal(), properties.getDefaultMaxPerRoute(),
                properties.getConnectionRequestTimeout(), properties.getConnectTimeout(), properties.getSocketTimeout());
    }

    /**
     * 盘古调这个方法
     * @param request 请求对象
     * @param feedback 命中后的回调
     * @return
     * @throws RtaRequestException
     */
    @Override
    public boolean isTarget(RtaRequest request, Consumer<Map<Object, Object>> feedback) throws RtaRequestException {
        HttpRequestBase requestBase = null;
        RtaType rtaType = getRtaType();
        try {
            requestBase = create(request);
            if (log.isDebugEnabled()) {
                log.debug("request: {}", requestBase.getURI());
            }

            HttpResponse response = httpClient.execute(requestBase);
            int statusCode = response.getStatusLine().getStatusCode();

            if (statusCode != HttpStatus.SC_OK) {
                log.warn("{} - statusCode: {}", rtaType, statusCode);
            }

            byte[] bytes = EntityUtils.toByteArray(response.getEntity());
            boolean isTarget = isTarget(request, bytes,feedback);
            return isTarget;
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("{} - {}", rtaType, e.getMessage());
            }
            throw new RtaRequestException(e);
        } finally {
            if (Objects.nonNull(requestBase) && !requestBase.isAborted()) {
                requestBase.abort();
            }
        }
    }

    /**
     * 创建请求
     * @param request
     * @return
     */
    private HttpRequestBase create(RtaRequest request) {
        if (StringUtil.isBlank(request.getExt())) {
            return null;
        }
        BidRequest bidreq = JsonHelper.convert(request.getExt(), BidRequest.class);
        SurgeBidding.BidRequest bidRequest = QiHangRequestBuilder.buildRequest(bidreq);
        HttpPost post = new HttpPost(properties.getUrl());
        post.setEntity(new ByteArrayEntity(bidRequest.toByteArray()));
        post.addHeader(HttpHeaders.CONTENT_TYPE, "application/x-protobuf;charset=UTF-8");
        return post;
    }

    /**
     * 判断返回值
     * @param request
     * @param bytes
     * @param feedback
     * @return
     */
    private boolean isTarget(RtaRequest request, byte[] bytes, Consumer<Map<Object, Object>> feedback) {
        try {
            SurgeBidding.BidResponse qiHangResponse = SurgeBidding.BidResponse.parseFrom(bytes);
            List<String> nurls = qiHangResponse.getSeatBidList().stream()
                    .flatMap(seatBid -> seatBid.getBidList().stream())
                    .map(SurgeBidding.BidResponse.Bid::getNurl)
                    .filter(StringUtils::isNotBlank)
                    .collect(Collectors.toList());

            if (!nurls.isEmpty() && feedback != null) {
                Map<Object, Object> res = new HashMap<>();
                res.put("nurls", nurls);
                res.put("price",qiHangResponse.getSeatBid(0).getBid(0).getPrice());
                feedback.accept(res);
            }
            return !qiHangResponse.getSeatBidList().isEmpty();
        } catch (InvalidProtocolBufferException e) {
            log.error("qiHang buildResponse error:", e);
            return false;
        }
    }


    @Override
    public RtaType getRtaType() {
        return RtaType.QiHangRtb;
    }

    @Override
    public RtaClientProperties getProperties() {
        return properties;
    }
}
