package com.bxm.pangu.rta.common.qihang.request;

import com.bxm.user.id.generator.DeviceHelper;
import com.bxm.user.id.generator.DeviceInfo;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;

import java.io.Serializable;
import java.util.ArrayList;

/**
 * 用户设备信息
 * @author allen
 * @see com.bxm.pangu.common.bidding.initializer.DeviceInitializer
 */
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class PanGuDevice implements Serializable {
    /**
     * IP
     */
    private String ip;
    /**
     * User-Agent
     */
    private String ua;
    /**
     * 运营商（电信/移动/联通）
     * @see com.bxm.pangu.common.constant.bid.CarrierType
     */
    private Long carrier;
    /**
     * 网络类型
     * @see com.bxm.pangu.common.constant.bid.ConnectionType
     */
    private Integer connection_type;
    /**
     * 品牌
     */
    private String brand;
    /**
     * 制造商
     */
    private String make;
    /**
     * 机型
     */
    private String model;
    /**
     * 操作系统类型
     * @see Os
     */
    private String os;
    /**
     * 操作系统版本号
     */
    private String osv;

    /**
     * imei
     */
    private String imei;
    private String imei_md5;
    /**
     * idfa
     */
    private String idfa;
    private String idfa_md5;
    /**
     * AndroidId
     */
    private String androidid;
    private String androidid_md5;
    /**
     * OAID
     */
    private String oaid;
    private String oaid_md5;
    /**
     * Mac address
     */
    private String mac;
    private String mac_md5;
    /**
     * Google Advertising ID,谷歌广告标识码
     */
    private String gaid;
    private String gaid_md5;
    /**
     * 设备屏幕
     */
    private Screen screen;
    /**
     * 设备类型
     */
    private String device_type;
    /**
     * 用户标识是否不为空。
     * @return 从 8 个用户标识里判断，只要有 1 个标识不为空则返回 {@code true}
     */
    public boolean isNotEmptyOfIdentity() {
        ArrayList<String> strings = Lists.newArrayList(imei, imei_md5, idfa, idfa_md5, androidid, androidid_md5, oaid, oaid_md5);
        return strings.stream().anyMatch(StringUtils::isNotBlank);
    }

    /**
     * 获取 UID，从 8 个用户标识属性里按顺序查找，找到即返回。
     *
     * <a href="https://bxmrds.yuque.com/staff-pe8atq/qwbihw/25989643">参考规范</a>
     * @return UID
     * @see #getUid()
     * @see #getOriginalUid(String...)
     */
    public String getOriginalUid() {
        return DeviceHelper.getUid(new DeviceInfo()
                .setImei(imei)
                .setImeiMd5(imei_md5)
                .setAnid(androidid)
                .setAnidMd5(androidid_md5)
                .setOaid(oaid)
                .setOaidMd5(oaid_md5)
                .setIdfa(idfa)
                .setIdfaMd5(idfa_md5));
    }

    /**
     * 获取 UID，从指定传入的 {@code original} 数组中按顺序查找，找到即返回。
     * @param original 查找数组
     * @return UID
     * @see #getOriginalUid()
     */
    public String getOriginalUid(String...original) {
        for (String s1 : original) {
            if (StringUtils.isNotBlank(s1)) {
                return s1;
            }
        }
        return null;
    }

    /**
     * 获取 安全加密的 UID，从 8 个用户标识属性里按顺序查找，找到即返回。
     * 注意如果 imei / androidid / oaid / idfa 有值那么会进行 md5 加密。
     *
     * <a href="https://bxmrds.yuque.com/staff-pe8atq/qwbihw/25989643">参考规范</a>
     * @return UID
     * @see #getOriginalUid()
     */
    public String getUid() {
        return DeviceHelper.getUid(new DeviceInfo()
                .setImei(getImeiForMd5())
                .setImeiMd5(getImei_md5())
                .setAnid(getAndroididForMd5())
                .setAnidMd5(getAndroidid_md5())
                .setOaid(getOaidForMd5())
                .setOaidMd5(getOaid_md5())
                .setIdfa(getIdfaForMd5())
                .setIdfaMd5(getIdfa_md5()));
    }

    /**
     * 获取 imei 的MD5，如果 imei_md5 值不存在则取 {@link #getImeiForMd5()}
     * @return imei md5，可能为空。
     */
    public String getImei_md5() {
        String imeiMd5 = StringUtils.isNotBlank(imei_md5) ? imei_md5 : getImeiForMd5();
        if (StringUtils.isNotBlank(imeiMd5) && StringUtils.isBlank(imei_md5)) {
            this.imei_md5 = imeiMd5;
        }
        return imeiMd5;
    }

    /**
     * 获取 idfa 的MD5，如果 idfa_md5 值不存在则取 {@link #getIdfaForMd5()}
     * @return idfa md5，可能为空。
     */
    public String getIdfa_md5() {
        String idfaMd5 = StringUtils.isNotBlank(idfa_md5) ? idfa_md5 : getIdfaForMd5();
        if (StringUtils.isNotBlank(idfaMd5) && StringUtils.isBlank(idfa_md5)) {
            this.idfa_md5 = idfaMd5;
        }
        return idfaMd5;
    }

    /**
     * 获取 androidid 的MD5，如果 androidid_md5 值不存在则取 {@link #getAndroididForMd5()}
     * @return androidid md5，可能为空
     */
    public String getAndroidid_md5() {
        String androididMd5 = StringUtils.isNotBlank(androidid_md5) ? androidid_md5 : getAndroididForMd5();
        if (StringUtils.isNotBlank(androididMd5) && StringUtils.isBlank(androidid_md5)) {
            this.androidid_md5 = androididMd5;
        }
        return androididMd5;
    }

    /**
     * 获取 oaid 的MD5，如果 oaid_md5 值不存在则取 {@link #getOaidForMd5()}
     * @return oaid md5，可能为空
     */
    public String getOaid_md5() {
        String oaidMd5 = StringUtils.isNotBlank(oaid_md5) ? oaid_md5 : getOaidForMd5();
        if (StringUtils.isNotBlank(oaidMd5) && StringUtils.isBlank(oaid_md5)) {
            this.oaid_md5 = oaidMd5;
        }
        return oaidMd5;
    }

    public String getMac_md5() {
        String macMd5 = StringUtils.isNotBlank(mac_md5) ? mac_md5 : getMacForMd5();
        if (StringUtils.isNotBlank(macMd5) && StringUtils.isBlank(mac_md5)) {
            this.mac_md5 = macMd5;
        }
        return macMd5;
    }

    /**
     * 获取 gaid 的MD5，如果 gaid_md5 值不存在则取 {@link #getGaidForMd5()}
     * @return gaid md5，可能为空。
     */
    public String getGaid_md5() {
        String gaidMd5 = StringUtils.isNotBlank(gaid_md5) ? gaid_md5 : getGaidForMd5();
        if (StringUtils.isNotBlank(gaidMd5) && StringUtils.isBlank(gaid_md5)) {
            this.gaid_md5 = gaidMd5;
        }
        return gaidMd5;
    }

    public String getImeiForMd5() {
        return StringUtils.isNotBlank(imei) ? DigestUtils.md5Hex(imei) : imei;
    }

    public String getIdfaForMd5() {
        return StringUtils.isNotBlank(idfa) ? DigestUtils.md5Hex(idfa) : idfa;
    }

    public String getAndroididForMd5() {
        return StringUtils.isNotBlank(androidid) ? DigestUtils.md5Hex(androidid) : androidid;
    }

    public String getOaidForMd5() {
        return StringUtils.isNotBlank(oaid) ? DigestUtils.md5Hex(oaid) : oaid;
    }

    public String getMacForMd5() {
        return StringUtils.isNotBlank(mac) ? DigestUtils.md5Hex(mac) : mac;
    }

    public String getGaidForMd5() {
        return StringUtils.isNotBlank(gaid) ? DigestUtils.md5Hex(gaid) : gaid;
    }
}
