package com.bxm.pangu.rta.common.dyds;

import com.bxm.pangu.rta.common.*;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UUIDHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.curator.shaded.com.google.common.collect.Lists;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;
import org.springframework.http.MediaType;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;

/**
 * 抖音电商
 * @author jingyu.li
 * @date 2024-03-20
 */
@Slf4j
public class DydsRtaClient extends AbstractHttpClientRtaClient {

    private final DydsRtaProperties properties;

    public DydsRtaClient(DydsRtaProperties properties) {
        super(properties);
        this.properties = properties;
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        String param = request.getParam();
        if (StringUtils.isBlank(param)) {
            throw new RtaRequestException("dydsRta param is empty!");
        }
        String[] split = param.split("\\|");
        String strategy_ids = split[0];
        String appId = split[1];

        boolean is_md5 = false;
        String device_fingerprint_id = null;
        //DFA=1，IMEI=2，ADNDROID_ID=3，GOOGLE_AID=4，
        Integer device_fingerprint_type = null;
        String idfa = request.getIdfa();
        String idfaMd5 = request.getIdfa_md5();
        String imei = request.getImei();
        String imeiMd5 = request.getImei_md5();
        String androidid = request.getAndroidid();
        String androididMd5 = request.getAndroidid_md5();
        String gaid = request.getGaid();
        String gaidMd5 = request.getGaid_md5();
        if (StringUtils.isNotBlank(idfa) || StringUtils.isNotBlank(idfaMd5)) {
            device_fingerprint_type = 1;
            if (StringUtils.isNotBlank(idfaMd5)) {
                is_md5 = true;
                device_fingerprint_id = idfaMd5;
            } else {
                device_fingerprint_id = idfa;
            }
        } else if (StringUtils.isNotBlank(imei) || StringUtils.isNotBlank(imeiMd5)) {
            device_fingerprint_type = 2;
            if (StringUtils.isNotBlank(imeiMd5)) {
                is_md5 = true;
                device_fingerprint_id = imeiMd5;
            } else {
                device_fingerprint_id = imei;
            }
        } else if (StringUtils.isNotBlank(androidid) || StringUtils.isNotBlank(androididMd5)) {
            device_fingerprint_type = 3;
            if (StringUtils.isNotBlank(androididMd5)) {
                is_md5 = true;
                device_fingerprint_id = androididMd5;
            } else {
                device_fingerprint_id = androidid;
            }
        } else if (StringUtils.isNotBlank(gaid) || StringUtils.isNotBlank(gaidMd5)) {
            device_fingerprint_type = 4;
            if (StringUtils.isNotBlank(gaidMd5)) {
                is_md5 = true;
                device_fingerprint_id = gaidMd5;
            } else {
                device_fingerprint_id = gaid;
            }
        }

        if (StringUtils.isBlank(device_fingerprint_id)) {
            throw new RtaRequestException("dydsRta device is empty!");
        }

        DydsRtaRequest.DeviceInfo deviceInfo = new DydsRtaRequest.DeviceInfo();
        deviceInfo.setDevice_fingerprint_id(device_fingerprint_id);
        deviceInfo.setDevice_fingerprint_type(device_fingerprint_type);

        DydsRtaRequest.RequestData data = new DydsRtaRequest.RequestData();
        data.setIs_md5(is_md5);
        data.setStrategy_ids(Lists.newArrayList(strategy_ids.split(",")));
        data.setDevice_finger_print_info(deviceInfo);

        DydsRtaRequest dydsRtaRequest = new DydsRtaRequest();
        dydsRtaRequest.setReq_id(UUIDHelper.generate());
        dydsRtaRequest.setApp_id(appId);
        dydsRtaRequest.setTimestamp(System.currentTimeMillis() / 1000);
        dydsRtaRequest.setSign_type("MD5");
        dydsRtaRequest.setData(JsonHelper.convert(data));
        dydsRtaRequest.setSign(getSign(dydsRtaRequest));

        System.out.println(JsonHelper.convert(dydsRtaRequest));

        HttpPost httpPost = new HttpPost(getProperties().getUrl());
        httpPost.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE);
        httpPost.setEntity(new ByteArrayEntity(JsonHelper.convert2bytes(dydsRtaRequest)));
        return httpPost;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        DydsRtaResponse convert = JsonHelper.convert(json, DydsRtaResponse.class);
        return convert != null && convert.getCode() == 0 &&
                convert.getData() != null && !CollectionUtils.isEmpty(convert.getData().getMatch_results()) &&
                Arrays.stream(request.getParam().split("\\|")[0].split(","))
                        .anyMatch(strategyName -> convert.getData().getMatch_results().stream()
                                .anyMatch(matchResult -> matchResult.getHit_status() == 1 &&
                                        StringUtils.equals(strategyName, matchResult.getStrategy_name())));
    }


    @Override
    public RtaType getRtaType() {
        return RtaType.Dyds;
    }

    private String getSign(DydsRtaRequest request) {
        try {
            String signStr = String.format("app_id=%s&data=%s&req_id=%s&timestamp=%d%s",
                    request.getApp_id(), request.getData(), request.getReq_id(),
                    request.getTimestamp(), properties.getSecureKey());
            MessageDigest md = MessageDigest.getInstance("MD5");
            byte[] md5 = md.digest(signStr.getBytes(StandardCharsets.UTF_8));
            StringBuilder sb = new StringBuilder();
            for (byte b : md5) {
                sb.append(String.format("%02x", b));
            }
            return sb.toString();
        } catch (NoSuchAlgorithmException e) {
            log.warn("dydsRtaSign error e",e);
            throw new RtaRequestException("dydsRtaSign error");
        }
    }
}
