package com.bxm.pangu.rta.common;

import com.bxm.pangu.rta.common.core.RequestEvent;
import com.bxm.pangu.rta.common.micrometer.RtaClientMicroMeter;
import com.bxm.warcar.cache.DataExtractor;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.TargetFactory;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.apache.commons.lang.math.RandomUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.util.ClassUtils;

import java.util.Set;

/**
 * @author allen
 * @date 2022-02-15
 * @since 1.0
 */
@Slf4j
@Aspect
@Order(Ordered.HIGHEST_PRECEDENCE)
public class CachingAspect {

    private final Fetcher fetcher;
    private final Set<String> enable;
    private final EventPark eventPark;
    private final RtaClientMicroMeter meter;

    public CachingAspect(Fetcher fetcher, Set<String> enable, EventPark eventPark, RtaClientMicroMeter meter) {
        this.fetcher = fetcher;
        this.enable = enable;
        this.eventPark = eventPark;
        this.meter = meter;
    }

    @Pointcut("this(com.bxm.pangu.rta.common.RtaClient) && execution(* isTarget(..))")
    public void pointcut() {}

    @Around("pointcut()")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        Object target = point.getTarget();
        if (!(target instanceof RtaClient)) {
            return point.proceed();
        }
        RtaClient client = (RtaClient) target;
        Object[] args = point.getArgs();

        if (ArrayUtils.isEmpty(args)) {
            return point.proceed();
        }
        Object arg = args[0];
        if (!(arg instanceof RtaRequest)) {
            return point.proceed();
        }
        RtaRequest request = (RtaRequest) arg;

        if (!enable.contains(ClassUtils.getUserClass(client).getName())) {
            eventPark.post(new RequestEvent(this, client, request));
            return point.proceed();
        }

        meter.incrementWithCaching(client.getRtaType());

        return fetcher.fetch(new TargetFactory<Boolean>()
                .keyGenerator(new KeyGenerator() {
                    @Override
                    public String generateKey() {
                        String simple = ToStringBuilder.reflectionToString(request, ToStringStyle.SIMPLE_STYLE);
                        // RTA:CACHE:{rtaType}:{param}:{md5(request)}
                        String key = KeyBuilder.build("RTA", "CACHE", client.getRtaType().getType(), DigestUtils.md5Hex(simple));
                        if (log.isDebugEnabled()) {
                            log.debug("{}", key);
                        }
                        return key;
                    }
                })
                .cls(Boolean.class)
                .expireTimeInSecond((int) DateHelper.getRemainSecondsOfToday() + RandomUtils.nextInt(3600))
                .dataExtractor(new DataExtractor<Boolean>() {
                    @Override
                    public Boolean extract() {
                        try {
                            eventPark.post(new RequestEvent(this, client, request));
                            return (boolean) point.proceed();
                        } catch (Throwable throwable) {
                            return false;
                        }
                    }
                })
                .build());
    }
}
