package com.bxm.pangu.rta.common.baidu;

import com.bxm.pangu.rta.common.AbstractHttpClientRtaClient;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.pangu.rta.common.RtaType;
import com.bxm.warcar.utils.JsonHelper;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Map;
import java.util.TreeMap;

/**
 * @author allen
 * @date 2021-10-13
 * @since 1.0
 */
@Slf4j
public class BaiduRtaClient extends AbstractHttpClientRtaClient {

    private final BaiduRtaProperties properties;

    public BaiduRtaClient(BaiduRtaProperties properties) {
        super(properties);
        this.properties = properties;
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        String url = properties.getUrl();

        MultiValueMap<String, String> params = new LinkedMultiValueMap<>();
        params.add("version", "1");
        params.add("channel", properties.getChannel());
        params.add("type", request.getParam());
        params.add("time", String.valueOf(System.currentTimeMillis() / 1000));
        String imeiMd5 = request.getImei_md5();
        if (StringUtils.isNotBlank(imeiMd5)) {
            params.add("imei_md5", imeiMd5);
        }
        String oaid = request.getOaid();
        if (StringUtils.isNotBlank(oaid)) {
            params.add("oaid", oaid);
        }
        String idfa = request.getIdfa();
        if (StringUtils.isNotBlank(idfa)) {
            params.add("idfa", idfa);
        }

        UriComponentsBuilder uriComponentsBuilder = UriComponentsBuilder.fromUriString(url).replaceQueryParams(params);

        String sign = signature(params);

        uriComponentsBuilder.replaceQueryParam("sign", sign);

        return new HttpGet(uriComponentsBuilder.build().toString());
    }

    private String signature(MultiValueMap<String, String> params) {
        TreeMap<String, String> treeMap = new TreeMap<>(params.toSingleValueMap());
        StringBuilder s = new StringBuilder();
        for (Map.Entry<String, String> entry : treeMap.entrySet()) {
            if (StringUtils.isNotBlank(entry.getValue())) {
                s.append(entry.getKey()).append("=").append(entry.getValue());
            }
        }
        s.append(properties.getToken());

        if (log.isDebugEnabled()) {
            log.debug("source string: {}", s);
        }

        return DigestUtils.md5Hex(s.toString()).toLowerCase();
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        return JsonHelper.convert(json, BaiduRtaResponse.class).isTarget();
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Baidu;
    }

    @Data
    private static class BaiduRtaResponse {

        private String errno;
        private String msg;
        private String data;

        public boolean isTarget() {
            return StringUtils.equals("1", data);
        }
    }
}
