package com.bxm.pangu.rta.scheduler.core;

import com.bxm.pangu.rta.common.RtaClient;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.pangu.rta.common.RtaRequestException;
import com.bxm.pangu.rta.scheduler.SchedulerProperties;
import com.bxm.pangu.rta.scheduler.core.download.OssControl;
import com.bxm.pangu.rta.scheduler.core.event.QueryLog;
import com.bxm.pangu.rta.scheduler.core.event.QueryLogEvent;
import com.bxm.pangu.rta.scheduler.core.event.QueryTargetEvent;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.RateLimiter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.util.ClassUtils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalTime;
import java.util.*;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * @author allen
 * @date 2021-12-20
 * @since 1.0
 */
@Slf4j
@Configuration
public abstract class AbstractFilesRtaQueryScheduler implements FileRtaQueryScheduler, DisposableBean, ApplicationListener<ApplicationStartedEvent> {

    private static final int DEFAULT_EXPIRE_TIME_IN_HOUR = 48;
    private static final int DEFAULT_EXPIRE_TIME_IN_SECONDS = DEFAULT_EXPIRE_TIME_IN_HOUR * 60 * 60;

    private final RateLimiter limiter;
    private final ThreadPoolExecutor executor;
    private final Consumer<RtaRequest> changeRequest;
    /**
     * RTA接口返回后，执行这个接口来动态获取人群包ID。
     * 该接口传入参数是具体 RTA 接口内部定义的数据。
     */
    private final Function<Map<Object, Object>, String> crowdPkgIdFetcher;
    /**
     * 过期时间获取器，单位：秒，如果没有实现则默认返回 {@link #DEFAULT_EXPIRE_TIME_IN_SECONDS}
     */
    private final Supplier<Integer> expireTimeFetcher;
    private SchedulerProperties properties;
    private EventPark eventPark;
    private OssControl ossControl;
    private AtomicInteger countDown = new AtomicInteger(0);

    private AtomicLong offset = new AtomicLong(0);

    /**
     * offset 存储调度器
     */
    private final ScheduledThreadPoolExecutor offsetSync = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("offset-sync"));

    public AbstractFilesRtaQueryScheduler(int corePoolSize, Consumer<RtaRequest> changeRequest, Function<Map<Object, Object>, String> crowdPkgIdFetcher) {
        this(corePoolSize, changeRequest, crowdPkgIdFetcher, null);
    }

    public AbstractFilesRtaQueryScheduler(int corePoolSize, Consumer<RtaRequest> changeRequest, Function<Map<Object, Object>, String> crowdPkgIdFetcher, Supplier<Integer> expireTimeFetcher) {
        this.changeRequest = changeRequest;
        this.limiter = RateLimiter.create(corePoolSize);
        this.crowdPkgIdFetcher = crowdPkgIdFetcher;
        this.expireTimeFetcher = Optional.ofNullable(expireTimeFetcher).orElseGet(() -> () -> DEFAULT_EXPIRE_TIME_IN_SECONDS);
        this.executor = new ThreadPoolExecutor(corePoolSize, corePoolSize, 0, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(), new NamedThreadFactory("rta-query"));
    }

    @Autowired
    public void setProperties(SchedulerProperties properties) {
        this.properties = properties;
    }

    @Autowired
    public void setEventPark(EventPark eventPark) {
        this.eventPark = eventPark;
    }

    @Autowired
    public void setOssControl(OssControl ossControl) {
        this.ossControl = ossControl;
    }

    @Override
    public void onApplicationEvent(ApplicationStartedEvent event) {
        this.offset = new AtomicLong(this.readOffset());
        this.offsetSync.scheduleWithFixedDelay(this::writeOffset, 0, 1, TimeUnit.MINUTES);
    }

    /**
     * 返回 RtaClient 实现。
     * @return 实例
     */
    protected abstract RtaClient getRtaClient();

    /**
     * 返回跳过的数量，即从返回值开始请求。
     * @return 默认：0
     */
    protected int getSkip() {
        return 0;
    }

    /**
     * 返回请求次数限制
     *
     * @return 限制次数
     */
    protected int getRequestTotalLimit() {
        return Integer.MAX_VALUE;
    }

    @Scheduled(cron = "0 0 0 * * ?")
    public void resetOffset() {
        offset = new AtomicLong(0);
        writeOffset();
        log.info("{} reset offset.", getSimpleName());
    }

    /**
     * 每天 1 点开始执行，每 10 分钟检查一次数据，直到 11 点检查终止。
     */
    @Override
    @Scheduled(cron = "0 0 1 * * ?")
    public synchronized void execute() {
        try {
            String simpleName = getSimpleName();
            if (properties.getEnable().contains(simpleName)) {
                execute0();
            }
        } catch (Exception e) {
            log.error("execute0: ", e);
        }
    }

    private void execute0() {
        String simpleName = getSimpleName();
        log.info("Scheduler {} starting...", simpleName);

        // 获取设备数据集
        boolean nonNull = Objects.nonNull(changeRequest);
        RtaClient rtaClient = getRtaClient();

        String ossId = getOssId();
        if (StringUtils.isBlank(ossId)) {
            log.warn("{} ossId must not blank!", simpleName);
            return;
        }

        Map<Type, List<File>> mapFile;

        // 自旋逻辑，每 10 分钟检查一次数据，直到 11 点检查终止。
        do {
            mapFile = ossControl.getMapFile(ossId);
            if (MapUtils.isNotEmpty(mapFile)) {
                break;
            }
            try {
                TimeUnit.MINUTES.sleep(1);
            } catch (InterruptedException ignored) {
            }
        } while (LocalTime.now().getHour() < 23);

        offset = new AtomicLong(readOffset());
        int total = computeValueSize(mapFile);
        countDown = new AtomicInteger(total);
        if (countDown.get() == 0) {
            return;
        }

        LocalDate start = LocalDate.now();
        int skip = getSkip();
        log.info("{} Skip to {}", simpleName, skip);
        int requestTotalLimit = getRequestTotalLimit() + skip;
        if (requestTotalLimit <= 0) {
            requestTotalLimit = Integer.MAX_VALUE;
        }

        if (log.isInfoEnabled()) {
            log.info("{} - Total size {} has been ready at {}.", simpleName, countDown.get(), start);
        }

        AtomicLong index = new AtomicLong(0);
        IterableStart: for (Map.Entry<Type, List<File>> entry : mapFile.entrySet()) {
            Type type = entry.getKey();
            List<File> files = entry.getValue();

            for (File file : files) {
                List<String> values = readFile(file);
                if (log.isInfoEnabled()) {
                    log.info("{} Starting read {} file {} the size is {}", simpleName, type, file, values.size());
                }

                for (String id : values) {
                    // 执行索引
                    long i = index.incrementAndGet();

                    // 跳过游标
                    long offset = this.offset.get();

                    // 执行次数
                    int times = total - this.countDown.decrementAndGet();

                    if (skip >= i || i <= skip + offset) {
                        continue;
                    }

                    this.limiter.acquire();

                    // now is tomorrow
                    if (times > requestTotalLimit || LocalDate.now().isAfter(start)) {
                        this.countDown.set(0);
                        if (log.isInfoEnabled()) {
                            log.info("{} Scheduled abort!", simpleName);
                        }
                        break IterableStart;
                    }

                    this.offset.incrementAndGet();

                    this.executor.submit(createNewTask(nonNull, rtaClient, type, id));
                }
            }
        }

        log.info("Scheduler {} was finished!", simpleName);

        System.gc();
    }

    private Runnable createNewTask(boolean nonNull, RtaClient rtaClient, Type type, String id) {
        return new Runnable() {
            @Override
            public void run() {
                try {
                    run0();
                } catch (Exception e) {
                    log.error("", e);
                }
            }

            private void run0() {
                RtaRequest request = new RtaRequest();

                if (type == Type.IMEI) {
                    request.setImei_md5(id);
                } else if (type == Type.OAID) {
                    request.setOaid_md5(id);
                }

                if (nonNull) {
                    changeRequest.accept(request);
                }

                try {
                    boolean isTarget = rtaClient.isTarget(request, new Consumer<Map<Object, Object>>() {
                        @Override
                        public void accept(Map<Object, Object> objectObjectMap) {
                            String thisCrowdPkgId = crowdPkgIdFetcher.apply(objectObjectMap);
                            int expireTime = expireTimeFetcher.get();
                            if (StringUtils.isNotBlank(thisCrowdPkgId)) {
                                eventPark.post(new QueryTargetEvent(this, type, id, thisCrowdPkgId, expireTime));
                                QueryLog queryLog = QueryLog.builder()
                                        .taskId(getSimpleName())
                                        .type(type)
                                        .id(id)
                                        .crowdPkgId(thisCrowdPkgId)
                                        .res(QueryLog.RES_SUCCESS)
                                        .build();
                                eventPark.post(new QueryLogEvent(this, queryLog));
                            } else {
                                log.warn("{} - No crowd package id.", getSimpleName());
                            }
                        }
                    });
                    if (!isTarget) {
                        QueryLog queryLog = QueryLog.builder()
                                .taskId(getSimpleName())
                                .type(type)
                                .id(id)
                                .res(QueryLog.RES_FAIL)
                                .build();
                        eventPark.post(new QueryLogEvent(this, queryLog));
                    }
                } catch (RtaRequestException e) {
                    QueryLog queryLog = QueryLog.builder()
                            .taskId(getSimpleName())
                            .type(type)
                            .id(id)
                            .res(QueryLog.RES_EX)
                            .exmsg(e.getMessage())
                            .build();
                    eventPark.post(new QueryLogEvent(this, queryLog));
                }
            }
        };
    }

    private List<String> readFile(File file) {
        try {
            return FileUtils.readLines(file, StandardCharsets.UTF_8);
        } catch (IOException e) {
            log.error("", e);
            return Lists.newArrayList();
        }
    }

    private int computeValueSize(Map<Type, List<File>> map) {
        int totalSize = 0;
        Collection<List<File>> values = map.values();
        if (CollectionUtils.isEmpty(values)) {
            throw new NullPointerException("values");
        }
        for (List<File> list : values) {
            for (File file : list) {
                try {
                    totalSize += FileUtils.readLines(file, StandardCharsets.UTF_8).size();
                } catch (IOException e) {
                    log.error("readLines: ", e);
                }
            }
        }
        return totalSize;
    }

    private long readOffset() {
        File file = offsetFile();
        long offset = 0;
        try {
            offset = NumberUtils.toLong(FileUtils.readFileToString(file, StandardCharsets.UTF_8), 0);
        } catch (FileNotFoundException ignored) {
        } catch (IOException e) {
            log.error("readFileToString: ", e);
        }
        if (offset != 0) {
            log.info("{} read offset {} on {}", getSimpleName(), offset, file);
        }
        return offset;
    }

    private void writeOffset() {
        try {
            File file = offsetFile();
            String l = TypeHelper.castToString(offset.get());
            FileUtils.write(file, l, StandardCharsets.UTF_8);
            if (offset.get() != 0) {
                log.info("{} write offset {} to {}", getSimpleName(), l, file);
            }
        } catch (Exception e) {
            log.error("write: ", e);
        }
    }

    private File offsetFile() {
        String simpleName = getSimpleName();
        return new File(properties.getFileCacheDir() + File.separator + ".rta" + File.separator + "offset_" + simpleName);
    }

    private String getSimpleName() {
        return ClassUtils.getShortName(getClass());
    }

    @Override
    public void destroy() {
        executor.shutdownNow();
        writeOffset();
        offsetSync.shutdown();
    }

    @Override
    public int getCorePoolSize() {
        return executor.getCorePoolSize();
    }

    @Override
    public int getActiveCount() {
        return executor.getActiveCount();
    }

    @Override
    public int getQueueSize() {
        return executor.getQueue().size();
    }

    @Override
    public long getCountDown() {
        return countDown.get();
    }
}
