package com.bxm.pangu.rta.scheduler.core;

import com.bxm.pangu.rta.common.RtaClient;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.pangu.rta.common.RtaRequestException;
import com.bxm.pangu.rta.common.RtaType;
import com.bxm.pangu.rta.scheduler.core.event.QueryTargetEvent;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.util.concurrent.RateLimiter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;

import java.time.LocalDate;
import java.time.LocalTime;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

/**
 * @author allen
 * @date 2021-12-20
 * @since 1.0
 * @deprecated 内存消耗太大，建议使用：{@link com.bxm.pangu.rta.scheduler.core.AbstractFilesRtaQueryScheduler}
 */
@Slf4j
@Deprecated
public abstract class AbstractRtaQueryScheduler implements RtaQueryScheduler, DisposableBean {

    private final RateLimiter limiter;
    private final ThreadPoolExecutor executor;
    private final Consumer<RtaRequest> changeRequest;
    private final String crowdPkgId;
    private final Integer expireTimeInHour;
    private EventPark eventPark;
    private AtomicInteger countDown = new AtomicInteger(0);


    public AbstractRtaQueryScheduler(int corePoolSize, Consumer<RtaRequest> changeRequest, String crowdPkgId, Integer expireTimeInHour) {
        this.changeRequest = changeRequest;
        this.crowdPkgId = crowdPkgId;
        this.limiter = RateLimiter.create(corePoolSize);
        this.expireTimeInHour = expireTimeInHour;
        this.executor = new ThreadPoolExecutor(corePoolSize, corePoolSize, 0, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(), new NamedThreadFactory("P-" + crowdPkgId));
    }

    @Autowired
    public void setEventPark(EventPark eventPark) {
        this.eventPark = eventPark;
    }

    /**
     * 返回 设备号 列表。
     * @return 设备号
     */
    protected abstract Map<Type, List<String>> fetchDevices();

    /**
     * 返回 RtaClient 实现。
     * @return 实例
     */
    protected abstract RtaClient getRtaClient();

    /**
     * 返回请求次数限制
     *
     * @return 限制次数
     */
    protected int getRequestTotalLimit() {
        return Integer.MAX_VALUE;
    }

    /**
     * 每天 1 点开始执行，每 10 分钟检查一次数据，直到 11 点检查终止。
     */
    @Override
    @Scheduled(cron = "0 0 1 * * ?")
    public synchronized void execute() {
        log.info("Scheduler {} starting...", getClass().getSimpleName());

        // 获取设备数据集
        boolean nonNull = Objects.nonNull(changeRequest);
        RtaClient rtaClient = getRtaClient();

        // 自旋逻辑，每 10 分钟检查一次数据，直到 11 点检查终止。
        Map<Type, List<String>> map;
        do {
            map = fetchDevices();
            if (MapUtils.isNotEmpty(map)) {
                break;
            }
            try {
                TimeUnit.MINUTES.sleep(10);
            } catch (InterruptedException ignored) {
            }
        } while (LocalTime.now().getHour() < 23);

        int total = computeValueSize(map);
        countDown = new AtomicInteger(total);
        if (countDown.get() == 0) {
            return;
        }

        LocalDate start = LocalDate.now();
        int requestTotalLimit = getRequestTotalLimit();

        RtaType rtaType = rtaClient.getRtaType();
        if (log.isInfoEnabled()) {
            log.info("{} - {} size has been ready at {}.", rtaType, countDown.get(), start);
        }

        map.forEach((type, values) -> {
            if (log.isInfoEnabled()) {
                log.info("Starting {} the size is {}", type, values.size());
            }

            for (String id : values) {
                limiter.acquire();

                // now is tomorrow
                int times = total - countDown.get();
                if (times >= requestTotalLimit || LocalDate.now().isAfter(start)) {
                    countDown.set(0);
                    if (log.isInfoEnabled()) {
                        log.info("{} Scheduled abort!", rtaType);
                    }
                    break;
                }

                executor.submit(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            run0();
                        } catch (Exception e) {
                            log.error("", e);
                        } finally {
                            countDown.decrementAndGet();
                        }
                    }

                    private void run0() {
                        RtaRequest request = new RtaRequest();

                        if (type == Type.IMEI) {
                            request.setImei_md5(id);
                        } else if (type == Type.OAID) {
                            request.setOaid_md5(id);
                        }

                        if (nonNull) {
                            changeRequest.accept(request);
                        }

                        try {
                            if (rtaClient.isTarget(request)) {
                                eventPark.post(new QueryTargetEvent(this, type, id, crowdPkgId, 48 * 60 * 60));
                            }
                        } catch (RtaRequestException e) {
                            log.warn("isTarget: {}", e.getMessage());
                        }
                    }
                });
            }
        });

        log.info("{} was finished!", rtaType);

        map.clear();
        System.gc();
    }

    private int computeValueSize(Map<Type, List<String>> map) {
        int totalSize = 0;
        for (List<String> list : map.values()) {
            totalSize += list.size();
        }
        return totalSize;
    }

    @Override
    public void destroy() {
        executor.shutdownNow();
    }

    @Override
    public int getCorePoolSize() {
        return executor.getCorePoolSize();
    }

    @Override
    public int getActiveCount() {
        return executor.getActiveCount();
    }

    @Override
    public int getQueueSize() {
        return executor.getQueue().size();
    }

    @Override
    public long getCountDown() {
        return countDown.get();
    }
}
