package com.bxm.pangu.rta.scheduler.core.download;

import com.bxm.pangu.rta.scheduler.SchedulerProperties;
import com.bxm.pangu.rta.scheduler.core.FileRtaQueryScheduler;
import com.bxm.pangu.rta.scheduler.core.Type;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.utils.SafeMapHelper;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.util.ClassUtils;

import java.io.File;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * @author allen
 * @date 2022-01-17
 * @since 1.0
 */
@Slf4j
@Configuration
public class OssControl implements ApplicationListener<ApplicationStartedEvent>, DisposableBean {

    private final SchedulerProperties properties;
    private final Set<String> ossIds = Sets.newHashSet();
    private final ConcurrentHashMap<String, Map<Type, List<File>>> fileMap = new ConcurrentHashMap<>();
    private final ThreadPoolExecutor download = new ThreadPoolExecutor(20, 20, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(), new NamedThreadFactory("dl"));

    public OssControl(SchedulerProperties properties) {
        this.properties = properties;
    }

    @Scheduled(cron = "0 0 0 * * ?")
    public void clean() {
        clean0();
        download0();
    }

    @Override
    public void onApplicationEvent(ApplicationStartedEvent event) {
        Map<String, FileRtaQueryScheduler> beans = event.getApplicationContext().getBeansOfType(FileRtaQueryScheduler.class);
        beans.values().forEach(e -> {
            String ossId = e.getOssId();
            String shortName = ClassUtils.getShortName(e.getClass());
            if (properties.getEnable().contains(shortName)) {
                ossIds.add(ossId);
                log.info("Enabled {} for ossId: {}", shortName, ossId);
            }
        });

        clean0();
        download0();
    }

    @Override
    public void destroy() throws Exception {
        clean0();
    }

    public Map<Type, List<File>> getMapFile(String ossId) {
        return fileMap.get(ossId);
    }

    private void clean0() {
        this.fileMap.values().forEach(new Consumer<Map<Type, List<File>>>() {
            @Override
            public void accept(Map<Type, List<File>> typeListMap) {
                typeListMap.values().forEach(new Consumer<List<File>>() {
                    @Override
                    public void accept(List<File> files) {
                        files.forEach(file -> {
                            FileUtils.deleteQuietly(file);
                            log.info("Deleted: {}", file);
                        });
                    }
                });
            }
        });
        this.fileMap.clear();
    }

    private void download0() {
        int batchLines = properties.getBatchLines();
        ossIds.forEach(id -> {
            download.submit(new Runnable() {
                @Override
                public void run() {
                    OssDownload download = new OssDownload(properties, id);
                    download.downloads();
                    File oaidFile = download.getOaidFile();
                    Splitter splitter = new Splitter(oaidFile, batchLines, false);
                    List<File> oaidFiles = splitter.split();
                    log.info("{} - oaid total files num is: {}", id, oaidFiles.size());
                    File imeiFile = download.getImeiFile();
                    Splitter imeiSplitter = new Splitter(imeiFile, batchLines, false);
                    List<File> imeiFiles = imeiSplitter.split();
                    log.info("{} - imei total files num is: {}", id, imeiFiles.size());

                    SafeMapHelper.get(fileMap, id, new LinkedHashMap<>()).put(Type.OAID, oaidFiles);
                    SafeMapHelper.get(fileMap, id, new LinkedHashMap<>()).put(Type.IMEI, imeiFiles);
                }
            });
        });
    }
}
