package com.bxm.pangu.rta.scheduler.core.download;

import com.bxm.pangu.rta.scheduler.SchedulerProperties;
import com.bxm.pangu.rta.scheduler.core.FileDeviceFetcher;
import com.bxm.pangu.rta.scheduler.core.utils.DownloadHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;

import java.io.File;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author allen
 * @date 2022-01-17
 * @since 1.0
 */
@Slf4j
public class OssDownload {

    private final SchedulerProperties properties;
    private final String ossId;

    private String oaidUri;
    private String imeiUri;

    private File oaidFile;
    private File imeiFile;

    public OssDownload(SchedulerProperties properties, String ossId) {
        this.properties = properties;
        this.ossId = ossId;

        String date = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy_MM_dd"));
        this.oaidUri = String.format(properties.getIdOaidFileUri(), ossId, date);
        this.imeiUri = String.format(properties.getIdImeiFileUri(), ossId, date);
    }

    public void downloads() {
        do {
            if (isReady()) {
                download();
                break;
            }

            log.info("{} not ready.", ossId);
            try {
                TimeUnit.MINUTES.sleep(10);
            } catch (InterruptedException ignored) {
            }
        } while (LocalTime.now().getHour() < 23);

        log.info("{} Download finished!", ossId);
    }

    private boolean isReady() {
        String fileCacheDir = properties.getFileCacheDir();
        String oaidReadyFileUrl = FileDeviceFetcher.getReadyFileUrl(oaidUri);
        String imeiReadyFileUrl = FileDeviceFetcher.getReadyFileUrl(imeiUri);
        DownloadHelper oaidHelper = new DownloadHelper(oaidReadyFileUrl, fileCacheDir);
        File oaidFile = oaidHelper.download();
        if (Objects.isNull(oaidFile)) {
            return false;
        }
        FileUtils.deleteQuietly(oaidFile);
        DownloadHelper imeiHelper = new DownloadHelper(imeiReadyFileUrl, fileCacheDir);
        File imeiFile = imeiHelper.download();
        if (Objects.isNull(imeiFile)) {
            return false;
        }
        FileUtils.deleteQuietly(imeiFile);
        return true;
    }

    private void download() {
        String fileCacheDir = properties.getFileCacheDir();
        DownloadHelper oaidDownloadHelper = new DownloadHelper(oaidUri, fileCacheDir);
        DownloadHelper imeiDownloadHelper = new DownloadHelper(imeiUri, fileCacheDir);
        this.oaidFile = oaidDownloadHelper.download();
        this.imeiFile = imeiDownloadHelper.download();
    }

    public String getOaidUri() {
        return oaidUri;
    }

    public String getImeiUri() {
        return imeiUri;
    }

    public File getOaidFile() {
        return oaidFile;
    }

    public File getImeiFile() {
        return imeiFile;
    }
}
