package com.bxm.pangu.rta.scheduler.core.listener;

import com.bxm.pangu.rta.scheduler.SchedulerProperties;
import com.bxm.pangu.rta.scheduler.core.event.QueryLog;
import com.bxm.pangu.rta.scheduler.core.event.QueryLogEvent;
import com.bxm.pangu.rta.scheduler.core.listener.file.LogFileUploader;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.batch.BatchBlockingQueue;
import com.bxm.warcar.utils.batch.BatchQueue;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.scheduling.annotation.Scheduled;

import java.io.File;
import java.io.IOException;
import java.net.Inet4Address;
import java.net.UnknownHostException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

/**
 * @author allen
 * @date 2022-02-08
 * @since 1.0
 */
@Slf4j
public class LogWriteToFileEventListener implements EventListener<QueryLogEvent>, ApplicationListener<ApplicationStartedEvent> {

    private final static String R = RandomStringUtils.randomAlphanumeric(8);
    private final LogFileUploader uploader;
    private final BatchQueue<String> batchQueue;
    private final File dir;

    public LogWriteToFileEventListener(LogFileUploader uploader, SchedulerProperties properties) {
        this.uploader = uploader;
        this.dir = this.createDir(properties);

        final String primaryKey = getPrimaryKey();
        this.batchQueue = new BatchBlockingQueue<>(properties.getWriteLogBatchSize(), new Consumer<List<String>>() {
            @Override
            public void accept(List<String> strings) {
                try {
                    String fileName = getCurrentlyFile(0, primaryKey);
                    File file = new File(fileName);
                    if (!file.exists()) {
                        if (!file.createNewFile()) {
                            throw new IOException("Cannot create file: " + file);
                        }
                    }
                    FileUtils.writeLines(file, strings, true);
                } catch (IOException e) {
                    log.error("writeLines: ", e);
                }
            }
        }, 60 * 1000);
    }

    @Scheduled(cron = "0 10 0 * * ?")
    public void scheduleUpload() {
        this.uploadFile();
    }

    @Override
    public void onApplicationEvent(ApplicationStartedEvent event) {
        this.uploadFile();
    }

    private void uploadFile() {
        String primaryKey = getPrimaryKey();
        String fileName = getCurrentlyFile(-1, primaryKey);
        File file = new File(fileName);
        if (!file.exists()) {
            log.info("{} does not exists.", file);
            return;
        }

        log.info("{} starting upload.", file);
        uploader.upload(file, true);
    }

    private String getCurrentlyFile(int plusDays, String primaryKey) {
        return dir + File.separator + now("yyyy_MM_dd", plusDays) + "_" + primaryKey + ".json";
    }

    private String now(String pattern) {
        return now(pattern, 0);
    }

    private String now(String pattern, int plusDays) {
        return LocalDate.now().plusDays(plusDays).format(DateTimeFormatter.ofPattern(pattern));
    }

    @Override
    @Subscribe
    public void consume(QueryLogEvent event) {
        QueryLog queryLog = event.getQueryLog();
        Map<String, Object> body = new HashMap<>();
        body.put("stdate", now("yyyy-MM-dd"));
        body.put("taskid", queryLog.getTaskId());
        body.put("idtype", queryLog.getType());
        body.put("id", queryLog.getId());
        body.put("res", queryLog.getRes());
        body.put("exmsg", queryLog.getExmsg());
        body.put("crowdPkgId", queryLog.getCrowdPkgId());
        batchQueue.add(JsonHelper.convert(body));
    }

    private File createDir(SchedulerProperties properties) {
        String fileCacheDir = properties.getFileCacheDir() + File.separator + "logs";
        File file = new File(fileCacheDir);
        if (!file.exists()) {
            if (!file.mkdirs()) {
                throw new RuntimeException("Cannot create dir: " + file);
            }
        } else if (file.isFile()) {
            throw new RuntimeException(file + " is file!");
        }
        return file;
    }

    private static String getPrimaryKey() {
        try {
            String address = Inet4Address.getLocalHost().getHostAddress();
            if (StringUtils.isBlank(address)) {
                return R;
            }
            return address.replaceAll("\\.", "_");
        } catch (UnknownHostException e) {
            return R;
        }
    }
}
