package com.bxm.pangu.rta.scheduler.core.listener.file;

import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.model.*;
import com.aliyuncs.exceptions.ClientException;
import com.bxm.warcar.utils.StringHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

/**
 * @author allen
 * @date 2022-02-09
 * @since 1.0
 */
@Slf4j
@EnableConfigurationProperties(OssProperties.class)
public class OssLogFileUploader implements LogFileUploader {

    private final OssProperties properties;

    public OssLogFileUploader(OssProperties properties) {
        this.properties = properties;
    }

    @Override
    public void upload(File file) {
        this.upload(file, false);
    }

    @Override
    public void upload(File file, boolean deleteFileOnCompleted) {
        String endpoint = properties.getEndpoint();
        String accessKeyId = properties.getAccessKeyId();
        String accessKeySecret = properties.getAccessKeySecret();
        String bucketName = properties.getBucketName();
        OSS ossClient = new OSSClientBuilder().build(endpoint, accessKeyId, accessKeySecret);

        try {
            String fileName = "rta_log/" + LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd")) + "/" + file.getName();

            String readyFileName = fileName + ".ready";

            if (ossClient.doesObjectExist(bucketName, readyFileName)) {
                log.info("File {} was upload!", fileName);
                return;
            }

            uploadFile(file, bucketName, ossClient, fileName);
            upload(bucketName, ossClient, readyFileName);

            if (deleteFileOnCompleted) {
                FileUtils.deleteQuietly(file);
                log.info("Delete file: {}", file);
            }
        } catch (ClientException ce) {
            log.error("Error Message: " + ce.getMessage());
        } catch (Throwable oe) {
            log.error("", oe);
        } finally {
            ossClient.shutdown();
        }
    }

    private void upload(String bucketName, OSS ossClient, String fileName) {
        ByteArrayInputStream inputStream = new ByteArrayInputStream(StringHelper.convert(String.valueOf(System.currentTimeMillis())));
        PutObjectRequest request = new PutObjectRequest(bucketName, fileName, inputStream);
        PutObjectResult result = ossClient.putObject(request);
        log.info("Updated ready file: {}", result.getETag());
    }

    private void uploadFile(File file, String bucketName, OSS ossClient, String fileName) throws Throwable {
        log.info("Uploading: {}", fileName);

        UploadFileRequest uploadFileRequest = new UploadFileRequest(bucketName, fileName);
        // The local file to upload---it must exist.
        uploadFileRequest.setUploadFile(file.getPath());
        // Sets the concurrent upload task number to 5.
        uploadFileRequest.setTaskNum(5);
        // Sets the part size to 1MB.
        uploadFileRequest.setPartSize(1024 * 1024 * 10);
        // Enables the checkpoint file. By default it's off.
        uploadFileRequest.setEnableCheckpoint(true);

        UploadFileResult uploadResult = ossClient.uploadFile(uploadFileRequest);

        CompleteMultipartUploadResult multipartUploadResult = uploadResult.getMultipartUploadResult();

        log.info("Updated data file: {}", multipartUploadResult.getETag());
    }
}
