package com.bxm.pangu.rta.scheduler.core.utils;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.util.UriComponentsBuilder;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.List;

/**
 * @author allen
 * @date 2021-12-21
 * @since 1.0
 */
public class DownloadHelper {

    private static final Logger LOGGER = LoggerFactory.getLogger(DownloadHelper.class);

    private final String url;
    private final String fileCacheDir;
    private final File file;

    public DownloadHelper(String url, String fileCacheDir) {
        this.url = url;
        this.fileCacheDir = fileCacheDir;
        this.file = createFile(UriComponentsBuilder.fromUriString(url).build().getPathSegments());
    }

    public File download() {
        HttpURLConnection urlConnection = null;
        InputStream inputStream = null;
        FileOutputStream os = null;
        try {
            urlConnection = (HttpURLConnection) new URL(url).openConnection();
            int responseCode = urlConnection.getResponseCode();
            if (responseCode != 200) {
                return null;
            }
            inputStream = urlConnection.getInputStream();
            long contentLength = urlConnection.getHeaderFieldLong("Content-Length", -1);
            if (exists(contentLength)) {
                return file;
            } else {
                createFileIfNecessaryDelete();
            }
            long start = System.currentTimeMillis();
            if (LOGGER.isInfoEnabled()) {
                BigDecimal mb = toMb(contentLength);
                LOGGER.info("{} Starting download, The content length is {} MB...", url, mb);
            }
            float step = contentLength / 100;
            int cache = 10 * 1024 * 1024;
            byte[] buffer = new byte[contentLength > cache ? cache : (int) contentLength];

            os = new FileOutputStream(file, true);
            int len, i = 1;
            while ((len = inputStream.read(buffer)) != -1) {
                os.write(buffer, 0, len);
                long size = os.getChannel().size();
                if (step >= 1 && size >= step * i) {
                    LOGGER.info("Downloading progress\t{}%\t{} MB", i++, toMb(size));
                }
            }

            os.flush();

            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("{} Finished in {} ms!", url, (System.currentTimeMillis() - start));
            }

            return file;
        } catch (Exception e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("download: ", e);
            }
            return null;
        } finally {
            IOUtils.closeQuietly(inputStream);
            IOUtils.closeQuietly(os);
            if (null != urlConnection) {
                urlConnection.disconnect();
            }
        }
    }

    private boolean exists(long contentLength) {
        long length = file.length();
        return contentLength == length;
    }

    private File createFile(List<String> pathSegments) {
        StringBuilder s = new StringBuilder();
        s.append(fileCacheDir);
        for (String pathSegment : pathSegments) {
            s.append(File.separator).append(pathSegment);
        }
        File file = new File(s.toString());
        File parentFile = file.getParentFile();

        if (!parentFile.exists() && !parentFile.mkdirs()) {
            throw new RuntimeException("mkdir occur ex: " + parentFile);
        }

        if (!file.exists()) {
            createFile(file);
        }
        return file;
    }

    private void createFileIfNecessaryDelete() {
        if (file.exists()) {
            if (!file.delete()) {
                throw new RuntimeException("Cannot delete file: " + file);
            }
            LOGGER.info("Delete file: {}", file);
            createFile(file);
        }
    }

    private void createFile(File file) {
        try {
            if (!file.createNewFile()) {
                throw new RuntimeException("Cannot create file: " + file);
            }
            LOGGER.info("Created file: {}", file);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private BigDecimal toMb(long size) {
        return new BigDecimal(size).divide(new BigDecimal(1024 * 1024), 2, RoundingMode.HALF_UP);
    }
}
