package com.bxm.shop.interceptors.profit;

import com.bxm.shop.common.exception.RedisConstants;
import com.bxm.shop.common.exception.ResponseCodeType;
import com.bxm.shop.common.exception.ShopsException;
import com.bxm.shop.dal.Boost;
import com.bxm.shop.dal.UserDao;
import com.bxm.shop.dal.mapper.BoostMapper;
import com.bxm.shop.dal.mapper.OrderMapper;
import com.bxm.shop.dal.mapper.UserMapper;
import com.bxm.shop.integration.ShopManagerIntegration;
import com.bxm.shop.integration.config.PingduoduoConfig;
import com.bxm.shop.model.RebateConfig;
import com.bxm.shop.model.order.dao.OrderDao;
import com.bxm.shop.service.UserService;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.interceptor.AbstractInterceptor;
import com.bxm.warcar.integration.interceptor.Invocation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.concurrent.TimeUnit;

/**
 * @author yirenjie
 * createDate:  2018/12/24
 */
@Service
@Slf4j
public class FreeOrderInterceptor extends AbstractInterceptor {

    @Autowired
    @Qualifier("jedisUpdater")
    protected Updater updater;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Resource
    private PingduoduoConfig pingduoduoConfig;

    @Resource
    private BoostMapper boostMapper;

    @Resource
    private UserMapper userMapper;

    @Resource
    private UserService userService;

    @Resource
    private OrderMapper orderMapper;

    @Resource
    private ShopManagerIntegration shopManagerIntegration;

    /**
     * 业务处理
     *
     * @param invocation
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    protected void doIntercept(Invocation invocation) {
        ProfitRequestModel requestModel = (ProfitRequestModel) invocation.getRequestModel();
        OrderDao order = requestModel.getOrderDao();
        String freeKey = String.format(RedisConstants.HANDING_FREE_ORDER, order.getId());
        try {

            boolean hasSold = false;
            for (Integer status : pingduoduoConfig.getSold()) {
                if (status.equals(requestModel.getPddOrderInfo().getOrderStatus())) {
                    hasSold = true;
                    break;
                }
            }
            if (!hasSold) {
                return;
            }

            if (StringUtils.isNotBlank(stringRedisTemplate.opsForValue().getAndSet(freeKey, ""))) {
                String openid = requestModel.getOrderDao().getOpenid();

                UserDao user = userService.getUserByOpenid(openid);
                if (user.getAvailableFreeTimes() >= 1) {
                    user.setAvailableFreeTimes(user.getAvailableFreeTimes() - 1);
                    user.setFreeTimes(user.getFreeTimes() + 1);
                    userService.updateUser(user);
                    return;
                }

                Boost boost = boostMapper.findLeastByOpenid(openid);
                if (boost == null || boost.getStatus() != Boost.Status.S) {
                    log.warn("用户{}无0元购机会,订单{}订正为普通订单", openid, order.getId());
                    RebateConfig defaultRebate = shopManagerIntegration.getDefaultRebate();
                    requestModel.setRebateConfig(defaultRebate);
                    orderMapper.update2NormalOrder(order.getId(), defaultRebate);
                    order.setParentRate(defaultRebate.getPre());
                    order.setGrandparentRate(defaultRebate.getPre2());
                    order.setPurchaseRate(defaultRebate.getSelfPurchase());
                    order.setShareRate(defaultRebate.getShare());
                    order.setFinalPrice(defaultRebate.getFinalPrice());
                    return;
                }

                if (!boostMapper.push2UsedByOpenid(openid)) {
                    throw new ShopsException(ResponseCodeType.OPERATION_NOT_EXECUTED);
                }
                UserDao info = userMapper.findByOpenid(openid);
                info.setFreeTimes(info.getFreeTimes() + 1);
                if (!userService.updateUser(info)) {
                    throw new ShopsException(ResponseCodeType.OPERATION_NOT_EXECUTED);
                }
            }
        } catch (Exception e) {
            stringRedisTemplate.opsForValue().set(freeKey, "0", 7L, TimeUnit.DAYS);
            log.error("0元购订单拦截器发生异常", e);
            throw e;
        }

    }
}
