package com.bxm.shop.service.impl;

import com.bxm.shop.dal.mapper.OrderMapper;
import com.bxm.shop.dal.mapper.UserMapper;
import com.bxm.shop.dal.mapper.UserProfitMapper;
import com.bxm.shop.facade.model.common.Page;
import com.bxm.shop.facade.model.common.CommonDto;
import com.bxm.shop.facade.model.friend.FriendInfo;
import com.bxm.shop.facade.model.friend.FriendRelationInfo;
import com.bxm.shop.integration.config.PingduoduoConfig;
import com.bxm.shop.model.friend.vo.FriendCount;
import com.bxm.shop.model.friend.vo.OpenidAndAmount;
import com.bxm.shop.dal.UserDao;
import com.bxm.shop.service.FriendService;
import com.bxm.shop.utils.LogicalPageUtils;
import com.bxm.shop.utils.SortUtils;
import org.apache.commons.collections.CollectionUtils;
import org.dozer.Mapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

@Service
public class FriendServiceImpl implements FriendService {

    @Autowired
    private UserMapper userMapper;

    @Autowired
    private UserProfitMapper  userProfitMapper;

    @Autowired
    private PingduoduoConfig pingduoduoConfig;

    @Autowired
    private OrderMapper orderMapper;

    @Autowired
    private Mapper mapper;

    @Override
    public FriendInfo getInfo(String openid) {
        FriendInfo info = new FriendInfo();
        info.setDirectFriendNum(userMapper.countDirectFriendNum(openid));
        info.setIndirectFriendNum(userMapper.countIndirectFriendNum(openid));
        info.setContributeAmount(userProfitMapper.friendContributeAmount(openid,pingduoduoConfig.getSold()));
        if(info.getContributeAmount() == null){
            info.setContributeAmount(0L);
        }
        return info;
    }

    /**
     * 查询直接好友信息
     * @param dto
     * @return
     */
    @Override
    public Page<FriendRelationInfo> findDirectFriends(CommonDto dto) {
        Page page = new Page();
        page.setPageNum(dto.getPageNum());
        page.setPageSize(dto.getPageSize());
        List<UserDao> friends = userMapper.findDirectFriendInfo(dto.getOpenid());
        List<String> openids = getOpenids(friends);
        List<FriendRelationInfo> friendRelationInfos = transform(friends);
        List<OpenidAndAmount> opendIdAndAmounts = userProfitMapper.findDirectFriendContributeAmount(dto.getOpenid(), pingduoduoConfig.getSold());
        dealContributeAmount(friendRelationInfos, opendIdAndAmounts);
        if (CollectionUtils.isNotEmpty(openids)){
            List<OpenidAndAmount> userOrderAmount = orderMapper.findUserOrderAmount(openids, pingduoduoConfig.getSold());
            dealOrderAmount(friendRelationInfos, userOrderAmount);
        }
        SortUtils.sort(friendRelationInfos,"contributeAmount","desc");
        LogicalPageUtils.getPageList(page,friendRelationInfos);
        //统计好友数量
        List<FriendRelationInfo> pageList = (List<FriendRelationInfo>) page.getList();
        List<String> pageOpendIds = new ArrayList();
        if(CollectionUtils.isNotEmpty(pageList)){
            pageList.forEach(e ->{
                pageOpendIds.add(e.getOpenid());
            });
        }
        if(CollectionUtils.isNotEmpty(pageOpendIds)) {
            List<FriendCount> friendCount = userMapper.getFriendCount(pageOpendIds);
            dealFriendCount(pageList, friendCount);
        }
        return page;
    }

    /**
     * 查询间接好友信息
     * @param dto
     * @return
     */
    @Override
    public Page<FriendRelationInfo> findIndirectFriends(CommonDto dto) {
        Page page = new Page();
        page.setPageNum(dto.getPageNum());
        page.setPageSize(dto.getPageSize());
        List<UserDao> friends = userMapper.findIndirectFriendInfo(dto.getOpenid());
        List<String> openids = getOpenids(friends);
        List<FriendRelationInfo> friendRelationInfos = transform(friends);
        List<OpenidAndAmount> opendIdAndAmounts = userProfitMapper.findIndirectFriendContributeAmount(dto.getOpenid(), pingduoduoConfig.getSold());
        dealContributeAmount(friendRelationInfos,opendIdAndAmounts);
        if (CollectionUtils.isNotEmpty(openids)){
            List<OpenidAndAmount> userOrderAmount = orderMapper.findUserOrderAmount(openids, pingduoduoConfig.getSold());
            dealOrderAmount(friendRelationInfos, userOrderAmount);
        }
        SortUtils.sort(friendRelationInfos,"contributeAmount","desc");
        LogicalPageUtils.getPageList(page,friendRelationInfos);
        return page;
    }

    /**
     * 统计好友贡献金额
     * @param friendRelationInfos
     * @param opendIdAndAmounts
     */
    private void dealContributeAmount(List<FriendRelationInfo> friendRelationInfos, List<OpenidAndAmount> opendIdAndAmounts) {
        if(CollectionUtils.isNotEmpty(friendRelationInfos) && CollectionUtils.isNotEmpty(opendIdAndAmounts)){
            for(FriendRelationInfo friendRelationInfo:friendRelationInfos){
                for(OpenidAndAmount idAndAmount:opendIdAndAmounts){
                    if(friendRelationInfo.getOpenid().equals(idAndAmount.getOpenid())){
                        friendRelationInfo.setContributeAmount(idAndAmount.getAmount());
                    }
                }
            }
        }
    }

    /**
     * 统计好友订单金额
     * @param friendRelationInfos
     * @param friendCount
     */
    private void dealOrderAmount(List<FriendRelationInfo> friendRelationInfos, List<OpenidAndAmount> opendIdAndAmounts) {
        if(CollectionUtils.isNotEmpty(friendRelationInfos) && CollectionUtils.isNotEmpty(opendIdAndAmounts)){
            for(FriendRelationInfo friendRelationInfo:friendRelationInfos){
                for(OpenidAndAmount idAndAmount:opendIdAndAmounts){
                    if(friendRelationInfo.getOpenid().equals(idAndAmount.getOpenid())){
                        friendRelationInfo.setOrderAmount(idAndAmount.getAmount());
                    }
                }
            }
        }
    }

    /**
     * 统计好友数量
     * @param friendRelationInfos
     * @param friendCount
     */
    private void dealFriendCount(List<FriendRelationInfo> friendRelationInfos, List<FriendCount> friendCount ) {
        if(CollectionUtils.isNotEmpty(friendRelationInfos) && CollectionUtils.isNotEmpty(friendCount)){
            for(FriendRelationInfo friendRelationInfo:friendRelationInfos){
                for(FriendCount count:friendCount){
                    if(friendRelationInfo.getOpenid().equals(count.getOpenid())){
                        friendRelationInfo.setFriendCount(count.getCount());
                    }
                }
            }
        }
    }

    private List<String> getOpenids(List<UserDao> friends){
        List<String> openids = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(friends)){
            for(UserDao user:friends){
                openids.add(user.getOpenid());
            }
        }
        return openids;
    }

    private List<FriendRelationInfo> transform(List<UserDao> list){
        List<FriendRelationInfo> rs = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(list)){
            for(UserDao userDao:list){
                FriendRelationInfo info = new FriendRelationInfo();
                mapper.map(userDao,info);
                info.setContributeAmount(0L);
                info.setOrderAmount(0L);
                info.setFriendCount(0);
                rs.add(info);
            }
        }
        return rs;
    }
}
