package com.bxm.shop.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.shop.common.exception.RedisConstants;
import com.bxm.shop.common.exception.ResponseCodeType;
import com.bxm.shop.common.exception.ShopsException;
import com.bxm.shop.common.utils.CalculateUtils;
import com.bxm.shop.facade.model.common.Page;
import com.bxm.shop.facade.model.goods.GoodsDetailVo;
import com.bxm.shop.facade.model.goods.GoodsPoolVO;
import com.bxm.shop.facade.model.goods.GoodsQueryRo;
import com.bxm.shop.facade.model.goods.GoodsVo;
import com.bxm.shop.facade.model.goods.ThemeGoodsVo;
import com.bxm.shop.model.RebateConfig;
import com.bxm.shop.integration.ShopManagerIntegration;
import com.bxm.shop.integration.pdd.PddGoodsIntegration;
import com.bxm.shop.model.OrderType;
import com.bxm.shop.model.goods.vo.Good;
import com.bxm.shop.model.goods.vo.GoodDetail;
import com.bxm.shop.model.goods.vo.GoodsList;
import com.bxm.shop.service.GoodsService;
import com.bxm.shopmanager.facade.model.GoodsPoolDTO;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.dozer.Mapper;
import org.springframework.beans.BeanUtils;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.ResourceUtils;

import javax.annotation.Resource;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author yirenjie
 * createDate:  2018/11/8
 */
@Service
@Slf4j
public class GoodsServiceImpl implements GoodsService {

    @Resource
    private PddGoodsIntegration pddGoodsIntegration;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Resource
    private Mapper mapper;

    @Resource
    private ShopManagerIntegration shopManagerIntegration;

    @Override
    public Page<GoodsVo> getList(GoodsQueryRo goodsQueryRo) {
        Page<GoodsVo> page = new Page<>();
        page.setPageSize(goodsQueryRo.getPageSize());
        page.setPageNum(goodsQueryRo.getPageNum());
        List<Good> cacheGoods = new ArrayList<>();
        List<String> queryGoodsIds = Lists.newArrayList();
        List<String> goodsIdList = new ArrayList<>();   // 商品池id集合
        GoodsPoolDTO goodsPoolDTO = shopManagerIntegration.getGoodsPoolById(ShopManagerIntegration.GoodsPoolType.FREE.getId());
        List<String> freeGoodsIds = goodsPoolDTO == null ? Lists.newArrayList() : goodsPoolDTO.getGoodsIds();

        long size = 0;  // 商品池size
        if (goodsQueryRo.getTagId() == null) {  // 只有当查询分类为全部时(tag==null),才需要优先展示商品池配置商品
            GoodsPoolDTO topGoodsPool = shopManagerIntegration.getGoodsPoolById(ShopManagerIntegration.GoodsPoolType.TOP.getId());
            goodsIdList = goodsPoolDTO == null ? Lists.newArrayList() : topGoodsPool.getGoodsIds();
            size = goodsIdList.size();
        }
        long startIndex = (goodsQueryRo.getPageNum() - 1) * goodsQueryRo.getPageSize(); // 从0开始的
        if (size - (startIndex + 1) >= 0) {
            long exceptEndIndex = startIndex + goodsQueryRo.getPageSize();
            boolean enough = exceptEndIndex <= (size - 1);
            queryGoodsIds = goodsIdList.subList((int) startIndex, enough ? (int) exceptEndIndex : (int) size);
            cacheGoods = pddGoodsIntegration.queryGoodsList(queryGoodsIds);
            if (enough) {
                Map<String, RebateConfig> rebateConfigs = shopManagerIntegration.getRebateByIds(queryGoodsIds);
                page.setTotal(size + goodsQueryRo.getPageSize());   // 为防止前端以为是最后一页,所以在商品池数量上加一个分页数量
                page.setList(cacheGoods.stream().map(good -> trans2Vo(good, freeGoodsIds, rebateConfigs)).collect(Collectors.toList()));
                return page;
            }
        }
        int remain = (int) (size % goodsQueryRo.getPageSize());
        long offsetPage = size / goodsQueryRo.getPageSize();
        long pageNum = goodsQueryRo.getPageNum();
        goodsQueryRo.setPageNum((int) (pageNum - offsetPage));

        // 通过两次查询来组装本页数据
        ArrayList<Good> result = new ArrayList<>(goodsQueryRo.getPageSize());
        if (cacheGoods.isEmpty() && goodsQueryRo.getPageNum() > 1) { // 如果是第一页则略过本次查询
            goodsQueryRo.setPageNum(goodsQueryRo.getPageNum() - 1);
            List<Good> goods = pddGoodsIntegration.queryGoodsList(goodsQueryRo).getGoodsList();
            if (goods != null) {
                result.addAll(goods.subList(goods.size() - remain, goods.size()));
            }
            goodsQueryRo.setPageNum(goodsQueryRo.getPageNum() + 1);
        }
        GoodsList.GoodsSearchResponse response = pddGoodsIntegration.queryGoodsList(goodsQueryRo);
        List<Good> goods = response.getGoodsList();
        if (goods != null && !goods.isEmpty()) {
            result.addAll(goods.subList(0, goods.size() - remain));
        }

        // 组装后的数据处理
        List<String> resultIds = Lists.newArrayList();
        resultIds.addAll(queryGoodsIds);
        resultIds.addAll(result.stream().map(good -> good.getGoodsId().toString()).collect(Collectors.toList()));
        Map<String, RebateConfig> rebateConfigs = shopManagerIntegration.getRebateByIds(resultIds);
        List<GoodsVo> vos = new ArrayList<>(cacheGoods.size() + result.size());
        for (Good cacheGood : cacheGoods) {
            vos.add(trans2Vo(cacheGood, freeGoodsIds, rebateConfigs));
        }
        for (Good good : result) {
            if (!goodsIdList.contains(good.getGoodsId().toString())) {
                vos.add(trans2Vo(good, freeGoodsIds, rebateConfigs));
            }
        }
        page.setTotal((long) response.getTotalCount());
        page.setList(vos);
        return page;
    }

    @Override
    public GoodsDetailVo getDetail(String goodId) {
        if (goodId == null) {
            throw new ShopsException(ResponseCodeType.PARAM_ILLEGAL);
        }
        GoodDetail goodDetail = pddGoodsIntegration.queryGoodDetail(goodId);
        if (goodDetail == null) {
            return null;
        }

        GoodsPoolDTO goodsPoolDTO = shopManagerIntegration.getGoodsPoolById(ShopManagerIntegration.GoodsPoolType.FREE.getId());
        List<String> freeGoodsIds = goodsPoolDTO == null ? Lists.newArrayList() : goodsPoolDTO.getGoodsIds();
        Map<String, RebateConfig> rebateConfigs = shopManagerIntegration.getRebateByIds(Lists.newArrayList(goodId));
        GoodsDetailVo goodDetailVo = (GoodsDetailVo) trans2Vo(goodDetail, freeGoodsIds, rebateConfigs);
        goodDetailVo.setPops(getPops());

        return goodDetailVo;
    }

    @Override
    public Page<GoodsVo> search(GoodsQueryRo goodsQueryRo) {
        GoodsList.GoodsSearchResponse response = pddGoodsIntegration.queryGoodsList(goodsQueryRo);
        if (response.getGoodsList() == null || response.getGoodsList().isEmpty()) {
            return null;
        }
        // 查询0元购商品列表
        GoodsPoolDTO goodsPoolDTO = shopManagerIntegration.getGoodsPoolById(ShopManagerIntegration.GoodsPoolType.FREE.getId());
        List<String> freeGoodsIds = goodsPoolDTO == null ? Lists.newArrayList() : goodsPoolDTO.getGoodsIds();
        // 查询返佣返利配置
        List<String> goodsIds = response.getGoodsList().stream().map(good -> good.getGoodsId().toString()).collect(Collectors.toList());
        Map<String, RebateConfig> rebateConfigs = shopManagerIntegration.getRebateByIds(Lists.newArrayList(goodsIds));

        List<GoodsVo> list = response.getGoodsList().stream().map(good -> trans2Vo(good, freeGoodsIds, rebateConfigs)).collect(Collectors.toList());
        return new Page<>(list, goodsQueryRo.getPageNum(), goodsQueryRo.getPageSize(), (long) response.getTotalCount());
    }

    @Override
    public GoodsVo trans2Vo(Good good, List<String> freeGoodsIds, Map<String,RebateConfig> rebateConfigs) {
        GoodsVo goodsVo = new GoodsVo();
        goodsVo.setGoodsId(good.getGoodsId().toString());
        goodsVo.setGoodsName(good.getGoodsName());
        goodsVo.setGoodsThumbnailUrl(good.getGoodsThumbnailUrl());
        long price = good.getMinGroupPrice() - good.getCouponDiscount();
        goodsVo.setPrice(price);
        Double reappearance = CalculateUtils.divide(price * good.getPromotionRate(), 1000L).doubleValue();
        goodsVo.setCouponDiscount(good.getCouponDiscount());
        goodsVo.setSoldQuantity(good.getSoldQuantity());

        // 获取返佣配置
        if (rebateConfigs == null) {
            return goodsVo;
        }
        RebateConfig rebateConfigRate = rebateConfigs.get(good.getGoodsId().toString());
        if (rebateConfigRate == null) {
            log.error("返佣返利配置为空");
            return goodsVo;
        }
        if (rebateConfigRate.getSelfPurchase() != null) {
            goodsVo.setReappearance(CalculateUtils.divide(reappearance * rebateConfigRate.getSelfPurchase(), 100L).longValue());
        } else if (rebateConfigRate.getFinalPrice() != null) {
            long rebateAmount = price - rebateConfigRate.getFinalPrice();
            Long maxAmount = Long.valueOf(stringRedisTemplate.opsForValue().get(RedisConstants.FREE_MAX_AMOUNT));
            goodsVo.setReappearance(rebateAmount > maxAmount ? maxAmount : rebateAmount);
        }

        // 标记商品类型
        if (freeGoodsIds != null) {
            String type = freeGoodsIds.contains(good.getGoodsId() + "") ? OrderType.FREE.name() : OrderType.NORMAL.name();
            goodsVo.setType(type);
        }

        // 商品详情特殊处理
        if (good instanceof GoodDetail){
            GoodsDetailVo goodDetailVo = mapper.map(goodsVo, GoodsDetailVo.class);
            goodDetailVo.setShareAward(CalculateUtils.divide(reappearance * rebateConfigRate.getShare(), 100L).longValue());
            goodDetailVo.setGoodsGalleryUrls(good.getGoodsGalleryUrls());
            goodDetailVo.setGoodsDesc(good.getGoodsDesc());

            if (good.getCouponStartTime() != null) {
                LocalDateTime startTime = LocalDateTime.ofEpochSecond(good.getCouponStartTime(), 0, ZoneOffset.ofHours(8));
                goodDetailVo.setCouponStartTime(startTime.format(DateTimeFormatter.ofPattern("yyyy.MM.dd")));
            }

            if (good.getCouponEndTime() != null) {
                LocalDateTime endTime = LocalDateTime.ofEpochSecond(good.getCouponEndTime(), 0, ZoneOffset.ofHours(8));
                goodDetailVo.setCouponEndTime(endTime.format(DateTimeFormatter.ofPattern("yyyy.MM.dd")));
            }
            return goodDetailVo;
        }
        return goodsVo;
    }

    @Override
    public ThemeGoodsVo getThemeGoodsList(Long goodsPoolId) {
        GoodsPoolDTO goodsPoolDTO = shopManagerIntegration.getGoodsPoolById(goodsPoolId);
        ThemeGoodsVo themeGoodsVo = new ThemeGoodsVo();
        themeGoodsVo.setColor(goodsPoolDTO.getColor());
        themeGoodsVo.setImage(goodsPoolDTO.getImage());
        themeGoodsVo.setName(goodsPoolDTO.getName());
        if (goodsPoolDTO.getOtherId() == null) {
            log.error("商品池id={}的主题商品id为空", goodsPoolId);
            throw new ShopsException(ResponseCodeType.SYSTEM_ERROR);
        }
        List<Good> goods = pddGoodsIntegration.queryThemeGoodsList(Long.valueOf(goodsPoolDTO.getOtherId()));
        if (goods.isEmpty()) {
            return themeGoodsVo;
        }
        List<String> ids = goods.stream().map(good -> good.getGoodsId().toString()).collect(Collectors.toList());
        Map<String, RebateConfig> rebateConfigs = shopManagerIntegration.getRebateByIds(ids);
        themeGoodsVo.setGoods(goods.stream().map(good -> trans2Vo(good, null, rebateConfigs))
                .collect(Collectors.toList()));
        return themeGoodsVo;
    }

    @Override
    public GoodsPoolVO getGoodsPoolById(Long id) {
        GoodsPoolVO goodsPoolVO=new GoodsPoolVO();
        GoodsPoolDTO goodsPoolDTO = shopManagerIntegration.getGoodsPoolById(id);
        BeanUtils.copyProperties(goodsPoolDTO,goodsPoolVO);
        List<String> goodsIds = goodsPoolDTO.getGoodsIds();
        List<Good> goods = pddGoodsIntegration.queryGoodsList(goodsIds);
        Map<String, RebateConfig> rebateByIds = shopManagerIntegration.getRebateByIds(goodsIds);
        List<GoodsVo> vos = new ArrayList();
        for(Good good:goods){
            GoodsVo goodsVo = trans2Vo(good, null, rebateByIds);
            vos.add(goodsVo);
        }
        goodsPoolVO.setGoodsVos(vos);
        return goodsPoolVO;
    }

    @Override
    public List<GoodsDetailVo.Pop> getPops() {
        // 从redis中取弹窗信息
        Long size = stringRedisTemplate.opsForList().size(RedisConstants.POP_LIST_KEY);
        final long POP_SIZE = 16;
        if (size < POP_SIZE) {
            size = genInfo2Redis();
        }
        long startIndex = (long) (Math.random() * size);
        startIndex = size - POP_SIZE < startIndex ? size - POP_SIZE : startIndex;
        List<String> popStrings = stringRedisTemplate.opsForList().range(RedisConstants.POP_LIST_KEY, startIndex, startIndex + POP_SIZE);

        return popStrings.stream().map(popString ->
                JSONObject.parseObject(popString, GoodsDetailVo.Pop.class)).collect(Collectors.toList());
    }

    /**
     * 生成信息并推送至redis
     */
    private long genInfo2Redis() {
        List<String> pops = new ArrayList<>();
        BufferedReader br = null;
        try {
            File mockFile = ResourceUtils.getFile(ResourceUtils.CLASSPATH_URL_PREFIX + "mock.txt");

            br = new BufferedReader(new FileReader(mockFile));
            String line;
            while ((line = br.readLine()) != null) {
                pops.add(line);
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                    log.error("", e);
                }
            }
        }
        return stringRedisTemplate.opsForList().rightPushAll(RedisConstants.POP_LIST_KEY, pops);
    }
}
