package com.bxm.shop.service.impl;

import com.bxm.shop.common.enums.OrderTypeEnum;
import com.bxm.shop.common.exception.RedisConstants;
import com.bxm.shop.common.exception.ResponseCodeType;
import com.bxm.shop.common.exception.ShopsException;
import com.bxm.shop.common.utils.CustomParameterUtils;
import com.bxm.shop.dal.Boost;
import com.bxm.shop.dal.UserDao;
import com.bxm.shop.dal.mapper.BoostMapper;
import com.bxm.shop.dal.mapper.OrderDetailMapper;
import com.bxm.shop.dal.mapper.OrderMapper;
import com.bxm.shop.dal.mapper.UserShareMapper;
import com.bxm.shop.facade.model.common.Page;
import com.bxm.shop.facade.model.order.*;
import com.bxm.shop.model.RebateConfig;
import com.bxm.shop.integration.ShopManagerIntegration;
import com.bxm.shop.integration.config.PingduoduoConfig;
import com.bxm.shop.integration.pdd.CommonGoodsPromotionUrlIntegration;
import com.bxm.shop.integration.pdd.OrderIncrementGetIntegration;
import com.bxm.shop.interceptors.profit.ProfitInterceptorChain;
import com.bxm.shop.interceptors.profit.ProfitInterceptorInvocation;
import com.bxm.shop.interceptors.profit.ProfitRequestModel;
import com.bxm.shop.model.OrderType;
import com.bxm.shop.model.goods.vo.GoodsCount;
import com.bxm.shop.model.order.dao.OrderDao;
import com.bxm.shop.model.order.dao.OrderDetailDao;
import com.bxm.shop.model.user.dao.UserShareDao;
import com.bxm.shop.service.OrderService;
import com.bxm.shop.service.UserService;
import com.bxm.shop.utils.DistributedLock;
import com.bxm.shopmanager.facade.model.GoodsPoolDTO;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.MessageSender;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.dozer.Mapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
public class OrderServiceImpl implements OrderService {

    private static final String SPLIT = "|";

    @Autowired
    private OrderMapper orderMapper;

    @Autowired
    private OrderDetailMapper orderDetailMapper;

    @Autowired
    private Mapper mapper;

    @Autowired
    private CommonGoodsPromotionUrlIntegration commonGoodsPromotionUrlIntegration;

    @Autowired
    private OrderIncrementGetIntegration orderIncrementGetIntegration;

    @Autowired
    private ProfitInterceptorChain profitInterceptorChain;

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired
    @Qualifier("jedisUpdater")
    protected Updater updater;
    @Autowired
    private MessageSender messageSender;

    @Autowired
    private PingduoduoConfig pingduoduoConfig;

    @Autowired
    private UserShareMapper userShareMapper;

    @Resource
    private ShopManagerIntegration shopManagerIntegration;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Resource
    private BoostMapper boostMapper;

    @Resource
    private UserService userService;

    @Autowired
    private DistributedLock distributedLock;

    private ThreadPoolExecutor poolExecutor = new ThreadPoolExecutor(Runtime.getRuntime().availableProcessors()*8,
            Runtime.getRuntime().availableProcessors()*8,0L, TimeUnit.SECONDS,new LinkedBlockingDeque(),
            new ThreadFactoryBuilder().setNameFormat("order-deal-pool-%d").build()
    );

    @Override
    public OrderVo add(OrderDto dto) {

        //1、生成虚拟订单
        OrderDao order = new OrderDao();
        mapper.map(dto,order);
        RebateConfig rebateConfig = shopManagerIntegration.getRebateById(dto.getGoodsId());
        order.setParentRate(rebateConfig.getPre());
        order.setFinalPrice(rebateConfig.getFinalPrice());
        order.setShareRate(rebateConfig.getShare());
        order.setPurchaseRate(rebateConfig.getSelfPurchase());
        order.setGrandparentRate(rebateConfig.getPre2());
        // 如果是0元购商品,但是用户已经没有0元购机会可以使用则不创建订单
        GoodsPoolDTO goodsPoolDTO = shopManagerIntegration.getGoodsPoolById(ShopManagerIntegration.GoodsPoolType.FREE.getId());
        boolean contains = goodsPoolDTO.getGoodsIds().contains(dto.getGoodsId());
        order.setOrderType(contains ? OrderType.FREE : OrderType.NORMAL);
        if (order.getOrderType() == OrderType.FREE) {
            UserDao user = userService.getUserByOpenid(dto.getOpenid());
            Boost boost = boostMapper.findLeastByOpenid(dto.getOpenid());
            if (user.getAvailableFreeTimes() < 1 && (boost == null || boost.getStatus() != Boost.Status.S)) {
                throw new ShopsException(ResponseCodeType.HAS_NO_FREE_CHANCE);
            }
        }
        order.setCreateTime(new Date());
        //保存分享相关信息
        if(null != dto.getShareId() && dto.getShareId()>0){
            UserShareDao shareInfo = userShareMapper.findById(dto.getShareId());
            //不能自己分享给自己
            if(!StringUtils.equalsIgnoreCase(order.getOpenid(),shareInfo.getOpenid())) {
                order.setShareId(shareInfo.getId());
                order.setShareOpenid(shareInfo.getOpenid());
            }else{
                order.setShareId(null);
                order.setShareOpenid(null);
            }
        }
        orderMapper.save(order);
        if (order.getOrderType() == OrderType.FREE) {
            // 0元购在redis中保存
            String freeKey = String.format(RedisConstants.HANDING_FREE_ORDER, order.getId());
            stringRedisTemplate.opsForValue().set(freeKey, "0", 7L, TimeUnit.DAYS);
        }
        //2、生成推广链接
        OrderVo orderVo = commonGoodsPromotionUrlIntegration.generate(dto.getGoodsId(),
                CustomParameterUtils.generateCustomParameter(order.getId(),dto.getShareId(),dto.getOpenid()));
        //失败重试2次
        for(int i=0;i<2;i++){
            if(null == orderVo){
                orderVo = commonGoodsPromotionUrlIntegration.generate(dto.getGoodsId(),
                        CustomParameterUtils.generateCustomParameter(order.getId(),dto.getShareId(),dto.getOpenid()));
            }
        }

        if(null == orderVo){
            orderVo = new OrderVo();
            Message msg = new Message();
            msg.setContent("commonGoodsPromotionUrlIntegration.generate error 创建推广位出错 dto.getGoodsId() :"+order.getGoodsId()+
                    " customParameter:"+CustomParameterUtils.generateCustomParameter(order.getId(),dto.getShareId(),dto.getOpenid()));
            messageSender.send2(msg);
        }
        orderVo.setOpenid(dto.getOpenid());
        orderVo.setOrderId(order.getId());
        orderVo.setGoodsId(order.getGoodsId());
        return orderVo;
    }

    @Override
    public void deleteAbandonOrder(Date date) {
        orderMapper.deleteAbandonOrder(date);
    }

    @Override
    public void synchronizeOrder(OrderDto dto) {

        Long lastSynTime = fetcher.fetch(RedisConstants.Order.getOrderIncrementSynTime(), Long.class);
        //如果1秒内已经同步过，则不再同步
        if(null != lastSynTime && lastSynTime/1000 == System.currentTimeMillis()/1000){
            return ;
        }
        PddOrderList orderListInfo = orderIncrementGetIntegration.getIncrementOrder(dto, Boolean.TRUE);
        //处理查询到的订单
        if (orderListInfo != null) {
            doDealOrderList(orderListInfo.getOrderList());
            /**
             * 返回结果按更新时间倒序排列
             * 从最后一页开始查询
             */
            Long totalCount = orderListInfo.getTotalCount();
            Long totalPage = totalCount % dto.getPageSize() == 0 ? totalCount / dto.getPageSize() : totalCount / dto.getPageSize() + 1;
            dto.setPageNum(totalPage.intValue());
            while(totalPage>1&&dto.getPageNum()>0){
                //分页查询
                if(dto.getPageNum() ==1){
                    //最后 一次遍历判断订单数量是否变化，如果变化则所有遍历一遍，防止漏同步
                    orderListInfo = orderIncrementGetIntegration.getIncrementOrder(dto, Boolean.TRUE);
                    if (orderListInfo != null) {
                      Long lastCount  = orderListInfo.getTotalCount();
                      if(lastCount != totalCount){
                          totalCount = lastCount;
                          totalPage = totalCount % dto.getPageSize() == 0 ? totalCount / dto.getPageSize() : totalCount / dto.getPageSize() + 1;
                          dto.setPageNum(totalPage.intValue()+1); //最后有减一，此处多加1
                      }
                    }
                }else{
                    orderListInfo = orderIncrementGetIntegration.getIncrementOrder(dto, Boolean.FALSE);
                }
                if (orderListInfo != null) {
                    doDealOrderList(orderListInfo.getOrderList());
                }
                dto.setPageNum(dto.getPageNum()-1);
            }
        }
        updater.update(RedisConstants.Order.getOrderIncrementSynTime(),System.currentTimeMillis());
    }

    @Override
    public Map<String, Integer> statisticsSoldGoodsNum(List<String> goodsId) {
        Map<String,Integer> map = new HashMap<>();
        for(String id:goodsId){
            map.put(id,0);
        }
        List<GoodsCount> goodsCounts = orderMapper.statisticsSoldGoodsNum(pingduoduoConfig.getSold());
        for(GoodsCount count : goodsCounts){
            map.put(count.getGoodsId(),count.getNum());
        }
        return map;
    }

    @Override
    public Page<OrderSearchVo> find(OrderDto dto) {
        Page page = new Page();
        Integer[] orderStatuses = getOrderStatusByType(dto.getOrderType());
        if(null != orderStatuses){
            dto.setOrderStatuses(Lists.newArrayList(orderStatuses));
        }

        Long lastGenCount = orderMapper.lastGenerateOrder(dto.getOpenid(), DateUtils.addMinutes(new Date(), -1));
        //最近生成的虚拟订单，从拼多多同步订单信息
        //现在小程序打开商品详情就会生成虚拟订单
        if(lastGenCount>0){
            OrderDto orderDto =new OrderDto();
            //同步最近20分钟的数据
            orderDto.setStartUpdateTime(System.currentTimeMillis()/1000-60);
            orderDto.setEndUpdateTime(System.currentTimeMillis()/1000);
            orderDto.setPageSize(50);
            orderDto.setPageNum(1);
            synchronizeOrder(orderDto);
        }

        Long count = orderMapper.totalCount(dto);

        if(null == dto.getPageNum()){
            dto.setPageNum(1);
        }

        if(null == dto.getPageSize()){
            dto.setPageSize(10);
        }
        dto.setPageStart((dto.getPageNum()-1)*dto.getPageSize());
        List<OrderDao> list = orderMapper.query(dto);
        page.setList(transform(list));
        page.setTotal(count);
        return page;
    }

    private List<OrderSearchVo> transform(List<OrderDao> list){
        List<OrderSearchVo> rs = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(list)){
            for(OrderDao obj:list){
                OrderSearchVo vo = new OrderSearchVo();
                mapper.map(obj,vo);
                vo.setOrderId(obj.getId());
                vo.setOrderType(getTypeByOrderStatus(obj.getOrderStatus()));
                //计算返现金额
                if(obj.getPromotionAmount() != null && obj.getPurchaseRate() != null) {
                    vo.setRebateAmount(obj.getPromotionAmount() * obj.getPurchaseRate() / 100);
                }
                if (obj.getPromotionAmount() != null && obj.getFinalPrice() != null) {
                    long maxAmount = Long.valueOf(stringRedisTemplate.opsForValue().get(RedisConstants.FREE_MAX_AMOUNT));
                    long rebate = (long) (obj.getOrderAmount() * 1.0 / obj.getGoodsQuantity());
                    vo.setRebateAmount(rebate > maxAmount ? maxAmount : rebate);
                }
                rs.add(vo);
            }
        }
        return rs;
    }

    /**
     * 处理订单列表
     * @param list
     */
    private void doDealOrderList(List<PddOrderInfo> list){
        if(CollectionUtils.isNotEmpty(list)){
            for(PddOrderInfo order:list){
                poolExecutor.submit(new Runnable() {
                    @Override
                    public void run() {
                        String requestId = UUID.randomUUID().toString();
                        if (null == order || StringUtils.isBlank(order.getCustomParameters())) {
                            return;
                        }
                        try {
                            boolean success = distributedLock.tryGetDistributedLock(RedisConstants.Order.getOrderSyncLock(order.getCustomParameters()).generateKey(), requestId, 10000);
                            if(!success){
                                return ;
                            }
                            //根据订单最后更新时间和订单状态判断订单是否发生变化
                            String updateTimeAndStatus = fetcher.hfetch(RedisConstants.Order.getOrderLastUpdatetimeAndStatus(), order.getOrderSn(), String.class);
                            //订单有变化需要处理
                            if (null == updateTimeAndStatus) {
                                updateOrderInfo(order);
                                updateOrderDetailInfo(order);
                                //redis记录订单更新时间和状态，
                                updater.hupdate(RedisConstants.Order.getOrderLastUpdatetimeAndStatus(), order.getOrderSn(), getOrderLastUpdatetimeAndStatus(order));
                                doInterceptor(order);
                            } else if (!updateTimeAndStatus.equals(getOrderLastUpdatetimeAndStatus(order))) {
                                updateOrderInfo(order);
                                updateOrderDetailInfo(order);
                                updater.hupdate(RedisConstants.Order.getOrderLastUpdatetimeAndStatus(), order.getOrderSn(), getOrderLastUpdatetimeAndStatus(order));
                                doInterceptor(order);
                            }
                        } catch (Exception e){
                            log.error("处理订单orderSn={}发生异常:", order.getOrderSn(), e);
                        }finally {
                            distributedLock.releaseDistributedLock(RedisConstants.Order.getOrderSyncLock(order.getCustomParameters()).generateKey(),requestId);
                        }
                    }
                });
            }
        }
    }

    private void updateOrderDetailInfo(PddOrderInfo order) {
        //获取订单号
        Long orderId = CustomParameterUtils.getOrderId(order.getCustomParameters());
        OrderDetailDao detail = new OrderDetailDao();
        mapper.map(order,detail);
        detail.setId(orderId);
        int count = orderDetailMapper.update(detail);
        if(count == 0){
            insertOrderDetailInfo(order);
        }
    }

    /**
     * 处理收益逻辑
     * @param order
     */
    public void doInterceptor(PddOrderInfo order) {
        Long orderId = CustomParameterUtils.getOrderId(order.getCustomParameters());
        ProfitRequestModel requestModel = new ProfitRequestModel();
        OrderDao orderDao = orderMapper.findById(orderId);
        requestModel.setPddOrderInfo(order);
        //返利配置取生成虚拟订单时的返利配置
        RebateConfig rebateConfig = new RebateConfig();
        rebateConfig.setPre(orderDao.getParentRate());
        rebateConfig.setPre2(orderDao.getGrandparentRate());
        rebateConfig.setSelfPurchase(orderDao.getPurchaseRate());
        rebateConfig.setShare(orderDao.getShareRate());
        rebateConfig.setFinalPrice(orderDao.getFinalPrice());
        requestModel.setRebateConfig(rebateConfig);
        requestModel.setOrderDao(orderDao);
        ProfitInterceptorInvocation invocation = new ProfitInterceptorInvocation();
        invocation.setRequestModel(requestModel);
        profitInterceptorChain.intercept(invocation);
    }

    private void insertOrderDetailInfo(PddOrderInfo order) {
        //获取订单号
        Long orderId = CustomParameterUtils.getOrderId(order.getCustomParameters());
        OrderDetailDao detail = new OrderDetailDao();
        mapper.map(order,detail);
        detail.setId(orderId);
        orderDetailMapper.save(detail);
    }


    /**
     * 更新订单表状态
     * @param order
     */
    private void updateOrderInfo(PddOrderInfo order) {
        //获取订单号
        Long orderId = CustomParameterUtils.getOrderId(order.getCustomParameters());
        OrderDao orderDao = new OrderDao();
        mapper.map(order,orderDao);
        orderDao.setId(orderId);
        orderDao.setModifyTime(new Date());
        int count = orderMapper.update(orderDao);
        //更新数量为0时，找不到对应虚拟订单
        if(count == 0){
            Message msg = new Message();
            msg.setContent("virtual can  not be found customParameters :"+order.getCustomParameters());
            messageSender.send2(msg);
        }
    }


    private String getOrderLastUpdatetimeAndStatus(PddOrderInfo order){
        return order.getOrderModifyAt()+SPLIT+order.getOrderStatus();
    }


    /**
     * 根据订单类型 返回订单状态
     * @param type
     * @return
     */
    public Integer[] getOrderStatusByType(String type){
        if("all".equals(type)){
            return null;
        }

        if(OrderTypeEnum.INVALID.getCode().equals(type)){
            return pingduoduoConfig.getInvalid();
        }

        if(OrderTypeEnum.CASHBACKED.getCode().equals(type)){
            return pingduoduoConfig.getCashbacked();
        }

        if(OrderTypeEnum.CASHBACKING.getCode().equals(type)){
            return pingduoduoConfig.getCashbacking();
        }
        return new Integer[0];
    }

    /**
     * 根据订单状态返回订单类型
     * @param status
     * @return
     */
    public String getTypeByOrderStatus(Integer status){
        for(Integer s:pingduoduoConfig.getInvalid()){
            if(status == s){
                return OrderTypeEnum.INVALID.getCode();
            }
        }
        for(Integer s:pingduoduoConfig.getCashbacking()){
            if(status == s){
                return OrderTypeEnum.CASHBACKING.getCode();
            }
        }
        for(Integer s:pingduoduoConfig.getCashbacked()){
            if(status == s){
                return OrderTypeEnum.CASHBACKED.getCode();
            }
        }

        return OrderTypeEnum.INVALID.getCode();
    }
}
