package com.bxm.shop.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.shop.common.constants.WeChatConstants;
import com.bxm.shop.common.exception.RedisConstants;
import com.bxm.shop.common.utils.InviteCodeUtils;
import com.bxm.shop.common.utils.OkHttpUtils;
import com.bxm.shop.config.WechatConfig;
import com.bxm.shop.dal.UserDao;
import com.bxm.shop.dal.mapper.UserMapper;
import com.bxm.shop.facade.model.user.UserDto;
import com.bxm.shop.facade.model.user.UserVo;
import com.bxm.shop.model.user.dto.WechatLoginReturn;
import com.bxm.shop.service.UserService;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import org.apache.commons.lang.StringUtils;
import org.dozer.Mapper;
import org.dozer.loader.api.BeanMappingBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import static org.dozer.loader.api.TypeMappingOptions.mapEmptyString;
import static org.dozer.loader.api.TypeMappingOptions.mapNull;

@Service
@Transactional(rollbackFor = Exception.class,timeout = 30)
public class UserServiceImpl implements UserService {

    @Autowired
    private WechatConfig wechatConfig;

    @Autowired
    private UserMapper userMapper;

    @Autowired
    private Mapper mapper;

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired
    @Qualifier("jedisUpdater")
    protected Updater updater;

    @Resource
    private UserService userService;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Override
    public UserVo login(UserDto dto) {
        UserVo vo = new UserVo();
        WechatLoginReturn loginObj = wechartLogin(dto.getCode()) ;
        if(null == loginObj.getOpenid()){
            return vo;
        }
        vo.setOpenid(loginObj.getOpenid());
        vo.setUnionid(loginObj.getUnioinid());
        vo.setSessionId(loginObj.getSessionkey());
        //验证是否为新用户
        UserDao info = userService.getUserByOpenid(vo.getOpenid());
        if(null == info){
            vo.setNewUser(true);
            info = new UserDao();
            mapper.map(dto,info);
            info.setOpenid(vo.getOpenid());
            info.setUnionid(vo.getUnionid());
            //有邀请码时需要保留关联关系
            if(StringUtils.isNotBlank(dto.getInviteCode())) {
                UserDao parentUser = userMapper.findByInviteCode(dto.getInviteCode());
                if(null != parentUser){
                    info.setParentOpenid(parentUser.getOpenid());
                    info.setParentUnionid(parentUser.getUnionid());
                    info.setGrandparentOpenid(parentUser.getParentOpenid());
                    info.setGrandparentUnionid(parentUser.getGrandparentUnionid());
                }
            }
            userMapper.insertSelective(info);
            //生成邀请码
            String inviteCode = InviteCodeUtils.generate(info.getId());
            info.setInviteCode(inviteCode);
            info.setAvailableFreeTimes(Integer.valueOf(stringRedisTemplate.opsForValue().get(RedisConstants.INIT_FREE_TIMES)));
        }
        userService.updateUser(info);
        //保存用户session信息
        updater.update(RedisConstants.User.getSessionId(vo.getOpenid(),vo.getSessionId()),"",RedisConstants.User.SESSION_TIME);
        vo.setInviteCode(info.getInviteCode());
        vo.setMobile(info.getMobile());
        return vo;
    }

    @Override
    public boolean update(UserDto dto) {
        UserDao userDao = new UserDao();
        mapper.map(dto,userDao);
        return updateUser(userDao);
    }

    @Override
    public UserDao getUserByOpenid(String openid) {
        return fetcher.hfetch(RedisConstants.User.getUserInfo(), openid,
                () -> userMapper.findByOpenid(openid),
                UserDao.class, userExpireSec());
    }

    @Override
    public boolean updateUser(UserDao userDao) {
        UserDao info = userMapper.findByOpenid(userDao.getOpenid());
        mapper.map(userDao, info);  // null值不会复制,见com.bxm.shop.config.DozerBeanMapperConfig.mapper
        if (userMapper.updateByPrimaryKeySelective(info) != 1) {
            return false;
        }
        updater.hupdate(RedisConstants.User.getUserInfo(),info.getOpenid(),info, userExpireSec());
        return true;
    }

    /**
     * 微信登录
     * 通过返回对象的openid判断是否登录成功
     * @param code
     * @return
     */
    private WechatLoginReturn wechartLogin(String code){
        Map<String,String> params = new HashMap<>();
        params.put(WeChatConstants.Param.APP_ID,wechatConfig.getAppId());
        params.put(WeChatConstants.Param.SECRET,wechatConfig.getAppSecret());
        params.put(WeChatConstants.Param.JS_CDOE,code);
        params.put(WeChatConstants.Param.GRANT_TYPE,WeChatConstants.Param.AUTH_CODE);
        String response = null;
        try {
            response = OkHttpUtils.post(wechatConfig.getJscode2sessionUrl(),params);
        } catch (IOException e) {
        }
        WechatLoginReturn wechartLogin = new WechatLoginReturn();
        if(null != response) {
            JSONObject obj = JSON.parseObject(response);
            if(null != obj && obj.getString(WeChatConstants.Param.OPENID) != null){
                wechartLogin.setOpenid(obj.getString(WeChatConstants.Param.OPENID));
                wechartLogin.setUnioinid(obj.getString(WeChatConstants.Param.UNIONID));
                wechartLogin.setSessionkey(obj.getString(WeChatConstants.Param.SESSION_KEY));
            }
        }
        return wechartLogin;
    }

    private int userExpireSec() {
        return (int) (3600 * 2.5 + Math.random() * 3600); // 2.5 - 3.5h 浮动
    }


}
