package com.bxm.shop.utils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Transaction;
import redis.clients.jedis.exceptions.JedisException;

import java.util.Collections;
import java.util.List;
import java.util.UUID;
public class DistributedLock {
    private final JedisPool jedisPool;
    private static final String LOCK_SUCCESS = "OK";
    private static final String SET_IF_NOT_EXIST = "NX";
    private static final String SET_WITH_EXPIRE_TIME = "PX";
    private static final Long RELEASE_SUCCESS = 1L;

    public DistributedLock(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
    }


    /**
     * 尝试获取分布式锁
     * @param jedis Redis客户端
     * @param lockKey 锁
     * @param requestId 请求标识
     * @param expireTime 超期时间
     * @return 是否获取成功
     */
    public  boolean tryGetDistributedLock(String lockKey, String requestId, int expireTime) {
        // 获取连接
        Jedis jedis = jedisPool.getResource();
        String result = null;
        try {
            result  = jedis.set(lockKey, requestId, SET_IF_NOT_EXIST, SET_WITH_EXPIRE_TIME, expireTime);
        }finally {
            if(null != jedis){
                jedis.close();
            }
        }
        if (LOCK_SUCCESS.equals(result)) {
            return true;
        }

        return false;

    }

    public void lock(final String key, String requestId, int expireTime) {
        while (true) {
            if (tryGetDistributedLock(key, requestId, expireTime)) {
                return;
            } else {
                try {
                    Thread.sleep(500);
                } catch (final InterruptedException e) {
                    throw new RuntimeException("获取锁失败", e);
                }
            }
        }
    }

    /**
     * 释放分布式锁
     * @param jedis Redis客户端
     * @param lockKey 锁
     * @param requestId 请求标识
     * @return 是否释放成功
     */
    public boolean releaseDistributedLock(String lockKey, String requestId) {
        // 获取连接
        Jedis jedis = jedisPool.getResource();
        Object result = null;
        try {
            String script = "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";
            result = jedis.eval(script, Collections.singletonList(lockKey), Collections.singletonList(requestId));
        }finally {
            if(null != jedis){
                jedis.close();
            }
        }
        if (RELEASE_SUCCESS.equals(result)) {
            return true;
        }
        return false;

    }

//   注释不好的：原因见：http://www.importnew.com/27477.html
//    /**
//     * 加锁
//     * @param locaName  锁的key
//     * @param acquireTimeout  获取超时时间
//     * @param timeout   锁的超时时间
//     * @return 锁标识
//     */
//    public String lockWithTimeout(String locaName,  long acquireTimeout, long timeout) {
//        Jedis conn = null;
//        String retIdentifier = null;
//        try {
//            // 获取连接
//            conn = jedisPool.getResource();
//            // 随机生成一个value
//            String identifier = UUID.randomUUID().toString();
//            // 锁名，即key值
//            String lockKey = "lock:" + locaName;
//            // 超时时间，上锁后超过此时间则自动释放锁
//            int lockExpire = (int)(timeout / 1000);
//
//            // 获取锁的超时时间，超过这个时间则放弃获取锁
//            long end = System.currentTimeMillis() + acquireTimeout;
//            while (System.currentTimeMillis() < end) {
//                if (conn.setnx(lockKey, identifier) == 1) {
//                    conn.expire(lockKey, lockExpire);
//                    // 返回value值，用于释放锁时间确认
//                    retIdentifier = identifier;
//                    return retIdentifier;
//                }
//                // 返回-1代表key没有设置超时时间，为key设置一个超时时间
//                if (conn.ttl(lockKey) == -1) {
//                    conn.expire(lockKey, lockExpire);
//                }
//
//                try {
//                    Thread.sleep(10);
//                } catch (InterruptedException e) {
//                    Thread.currentThread().interrupt();
//                }
//            }
//        } catch (JedisException e) {
//            e.printStackTrace();
//        } finally {
//            if (conn != null) {
//                conn.close();
//            }
//        }
//        return retIdentifier;
//    }
//
//    /**
//     * 释放锁
//     * @param lockName 锁的key
//     * @param identifier    释放锁的标识
//     * @return
//     */
//    public boolean releaseLock(String lockName, String identifier) {
//        Jedis conn = null;
//        String lockKey = "lock:" + lockName;
//        boolean retFlag = false;
//        try {
//            conn = jedisPool.getResource();
//            while (true) {
//                // 监视lock，准备开始事务
//                conn.watch(lockKey);
//                // 通过前面返回的value值判断是不是该锁，若是该锁，则删除，释放锁
//                if (identifier.equals(conn.get(lockKey))) {
//                    Transaction transaction = conn.multi();
//                    transaction.del(lockKey);
//                    List<Object> results = transaction.exec();
//                    if (results == null) {
//                        continue;
//                    }
//                    retFlag = true;
//                }
//                conn.unwatch();
//                break;
//            }
//        } catch (JedisException e) {
//            e.printStackTrace();
//        } finally {
//            if (conn != null) {
//                conn.close();
//            }
//        }
//        return retFlag;
//    }
}
