package com.bxm.spider.deal.dal.service.impl;

import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.mapper.Wrapper;
import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.bxm.spider.cache.RedisClient;
import com.bxm.spider.cache.constant.IdKeyConstant;
import com.bxm.spider.constant.exception.BaseRunException;
import com.bxm.spider.constant.monitor.MonitorConstant;
import com.bxm.spider.constant.monitor.MonitorHelper;
import com.bxm.spider.constant.url.UrlTypeEnum;
import com.bxm.spider.deal.common.constant.ContentConstant;
import com.bxm.spider.deal.common.utils.StringHelp;
import com.bxm.spider.deal.dal.mapper.CommentMapper;
import com.bxm.spider.deal.dal.mapper.VideoMapper;
import com.bxm.spider.deal.dal.service.VideoService;
import com.bxm.spider.deal.facade.model.ContentDto;
import com.bxm.spider.deal.facade.model.ContentVo;
import com.bxm.spider.deal.model.dao.Comment;
import com.bxm.spider.deal.model.dao.Video;
import com.bxm.spider.deal.model.dto.VideoCommentDto;
import com.bxm.spider.deal.model.dto.VideoDto;
import com.bxm.spider.oss.constant.OssConstant;
import com.bxm.spider.oss.service.AliYunOssService;
import com.bxm.spider.utils.DateUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * <p>
 * 爬取小视频表 服务实现类
 * </p>
 *
 * @author JandMin
 * @since 2018-10-29
 */
@Service
public class VideoServiceImpl extends ServiceImpl<VideoMapper, Video> implements VideoService {
    private final Logger LOG = LoggerFactory.getLogger(VideoServiceImpl.class);

    @Autowired
    private CommentMapper    commentMapper;
    @Autowired
    private VideoMapper      videoMapper;
    @Autowired
    private RedisClient      redisClient;
    @Resource(name = "aliYunOssServiceImpl")
    private AliYunOssService aliYunOssService;

    @Override
    public Video findVideoByVideoId(String videoId) {
        Video video = new Video();
        video.setUrlCode(videoId.hashCode());
        return videoMapper.selectOne(video);
    }

    @Override
    public int saveVideo(VideoDto videoDto) {
        if(null == videoDto.getVideo_url() || StringUtils.isBlank(videoDto.getVideo_url())) {
            throw new BaseRunException("the detailUrl is null");
        }
        Video video = new Video();
        Long id = redisClient.hincrByOne(IdKeyConstant.getPrimaryId(),IdKeyConstant.FIELD_VIDEO);
        if(null == id){
            id = StringHelp.getUUIDHashCode(16);
        }
        videoDto.setId(id);
        video.setId(id);
        video.setUrlCode(videoDto.getVideo_id().hashCode());
        setVideoValue(video,videoDto);
        return videoMapper.insert(video);
    }

    /**
     * 从页面解析的数据中为视频赋值
     * @date: 2018/10/30
     * @param video
     * @param videoDto
     * @return void
     * @throws
     */
    private void setVideoValue(Video video, VideoDto videoDto) {
        video.setSerialNum(videoDto.getSerialNum());
        video.setModifyTime(new Date());
        if(StringUtils.isNotBlank(videoDto.getNickname())) {
            video.setNickname(videoDto.getNickname().trim());
        }
        if(StringUtils.isNotBlank(videoDto.getDesc())) {
            video.setTitle(videoDto.getDesc().trim());
        }
        if(StringUtils.isNotBlank(videoDto.getVideo_id())) {
            video.setVideoId(videoDto.getVideo_id());
        }
        if(StringUtils.isNotBlank(videoDto.getAvatar())) {
            try{
                String ossUrl = aliYunOssService.getOssUrl(videoDto.getAvatar().trim(),videoDto.getSerialNum(), OssConstant.IMG_SUFFIX_HEAD);
                if(StringUtils.isNotBlank(ossUrl)){
                    video.setAuthorImageUrl(ossUrl);
                }
            }catch (Exception e){
                e.printStackTrace();
            }
        }
        if(StringUtils.isNotBlank(videoDto.getVideo_img())) {
            try{
                String ossUrl = aliYunOssService.getOssUrl(videoDto.getVideo_img().trim(),videoDto.getSerialNum(),OssConstant.IMG_SUFFIX_VIDEO);
                if(StringUtils.isNotBlank(ossUrl)){
                    video.setVideoImageUrl(ossUrl);
                }
            }catch (Exception e){
                e.printStackTrace();
            }
        }
        // 视频比例
        if(StringUtils.isNotBlank(videoDto.getWidthRatio())){
            video.setWidthRatio(Double.valueOf(videoDto.getWidthRatio()));
        }
        if(StringUtils.isNotBlank(videoDto.getVideo_url())) {
            video.setVideoUrl(videoDto.getVideo_url());
        }
        if(StringUtils.isNotBlank(videoDto.getSource_url())) {
            video.setUrl(videoDto.getSource_url());
        }
        if(null != videoDto.getZan()) {
            video.setZan(videoDto.getZan());
        }
        if(StringUtils.isNotBlank(videoDto.getChannel())) {
            video.setChannel(videoDto.getChannel());
        }
        if(StringUtils.isNotBlank(videoDto.getSource())) {
            video.setSource(videoDto.getSource());
        }
    }

    @Override
    public int updateVideo(Video video, VideoDto videoDto) {
        setVideoValue(video,videoDto);
        video.setStatus(ContentConstant.STATUS_VALID);
        return videoMapper.updateById(video);
    }

    @Transactional(rollbackFor={ Exception.class, BaseRunException.class },timeout = 120)
    @Override
    public Long persistVideo(VideoDto videoDto) {
        long start = System.currentTimeMillis();
        Video video = findVideoByVideoId(videoDto.getVideo_id());
        if(null == video){
            //保存视频信息
            int i = this.saveVideo(videoDto);
            if(i != 1){
                throw new BaseRunException("save video failed" + videoDto);
            }
            List<VideoCommentDto> commentDtoList = videoDto.getComments();
            if(null != commentDtoList) {
                List<Comment> comments = new ArrayList<>();
                for (VideoCommentDto commentDto : commentDtoList) {
                    Comment comment = new Comment();
                    comment.setSourceId(videoDto.getId());
                    if (StringUtils.isNotBlank(videoDto.getImgUlrPrefix())) {
                        commentDto.setAvatar(videoDto.getImgUlrPrefix() + commentDto.getAvatar());
                    }
                    try {
                        String ossUrl = aliYunOssService.getOssUrl(commentDto.getAvatar(), videoDto.getSerialNum(),OssConstant.IMG_SUFFIX_HEAD);
                        if (StringUtils.isNotBlank(ossUrl)) {
                            comment.setImgUrl(ossUrl);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    comment.setNickname(commentDto.getNickname());
                    comment.setContent(commentDto.getText());
                    comment.setTime(commentDto.getCreate_time());
                    comment.setZan(commentDto.getZan());
                    comment.setModifyTime(new Date());
                    comments.add(comment);
                }
                if (null != comments && comments.size() > 0) {
                    i = commentMapper.insertBath(comments);
                }
            }
            LOG.info(MonitorConstant.MONITOR, "【保存小视频结束】url:{},id:{},nickname:{},spend time {}ms {}",videoDto.getVideo_url(),videoDto.getId(),videoDto.getNickname(),
                    (System.currentTimeMillis() - start), MonitorHelper.ofSuccessLog(MonitorConstant.SAVE_INSERT, videoDto.getSerialNum(), UrlTypeEnum.URL_DETAIL));
        } else if ( ContentConstant.STATUS_RENEWAL == video.getStatus().intValue()) {
            //修改视频信息
            int i = updateVideo(video,videoDto);
            LOG.info(MonitorConstant.MONITOR, "【修改小视频结束】url:{},id:{},nickname:{},spend time {}ms {}",videoDto.getVideo_url(),videoDto.getId(),videoDto.getNickname(),
                    (System.currentTimeMillis() - start), MonitorHelper.ofSuccessLog(MonitorConstant.SAVE_UPDATE, videoDto.getSerialNum(), UrlTypeEnum.URL_DETAIL));
        } else {
            LOG.info("【小视频已存在】url:{},id:{},nickname:{}",videoDto.getVideo_url(),videoDto.getId(),videoDto.getNickname());
        }
        return videoDto.getId();
    }

    @Override
    public List<ContentVo> findVideoList(ContentDto contentDto) {
        Wrapper<Video> wrapper = new EntityWrapper<>();
        if (StringUtils.isNotBlank(contentDto.getTitle())) {
            wrapper.like("title", contentDto.getTitle());
        }
        if (StringUtils.isNotBlank(contentDto.getStart())) {
            wrapper.ge("modify_time", contentDto.getStart());
        }
        List<Video> videos = this.selectList(wrapper);
        List<ContentVo> list = new ArrayList<>();
        if(null != videos){
            for(Video video : videos){
                ContentVo contentVo = new ContentVo();
                contentVo.setChannel(video.getChannel());
                contentVo.setContent(video.getVideoUrl());
                contentVo.setDateTime(DateUtils.formatDateTime(video.getModifyTime()));
                contentVo.setDeployTime(DateUtils.formatDateTime(video.getCreateTime()));
                contentVo.setId(video.getId());
                contentVo.setImgUrl(video.getVideoImageUrl());
                contentVo.setSource(video.getSource());
                contentVo.setStatus(video.getStatus());
                contentVo.setTitle(StringUtils.isBlank(video.getDesc()) ? video.getTitle() : video.getDesc());
                contentVo.setUrl(video.getUrl());
                list.add(contentVo);
            }
        }
        return list;
    }

}
