package com.bxm.spider.deal.dal.service.impl;

import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.mapper.Wrapper;
import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.bxm.spider.cache.constant.IdKeyConstant;
import com.bxm.spider.constant.exception.BaseRunException;
import com.bxm.spider.constant.monitor.MonitorConstant;
import com.bxm.spider.constant.monitor.MonitorHelper;
import com.bxm.spider.constant.processor.ProcessorParameter;
import com.bxm.spider.deal.common.constant.WechatConstant;
import com.bxm.spider.deal.dal.mapper.NewsWechatMapper;
import com.bxm.spider.deal.dal.service.NewsWeChatContentService;
import com.bxm.spider.deal.dal.service.NewsWeChatService;
import com.bxm.spider.deal.dal.utils.IdHelps;
import com.bxm.spider.deal.facade.model.ContentDto;
import com.bxm.spider.deal.facade.model.ContentVo;
import com.bxm.spider.deal.model.dao.NewsWechat;
import com.bxm.spider.deal.model.dao.NewsWechatContent;
import com.bxm.spider.utils.DateUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * <p>
 * 微信爬取网页内容表 服务实现类
 * </p>
 *
 * @author kk.xie
 * @since 2018-10-30
 */
@Service
public class NewsWeChatServiceImpl extends ServiceImpl<NewsWechatMapper, NewsWechat> implements NewsWeChatService {
    private static final Logger logger = LoggerFactory.getLogger(NewsWeChatServiceImpl.class);

    @Autowired
    private NewsWeChatContentService newsWechatContentService;
    @Autowired
    private NewsWechatMapper newsWechatMapper;

    @Autowired
    private IdHelps idHelps;

    @Override
    public NewsWechat findByPmTitle(String wechatPm, String title) {
        NewsWechat newsWechat= this.selectOne(new EntityWrapper()
                .eq("wechat_mp", wechatPm)
                .eq("title", title));
        return newsWechat;
    }

    @Override
    public boolean insertWechat(NewsWechat newsWechat) {
        return this.insert(newsWechat);
    }

    @Override
    public boolean updateWechatById(NewsWechat newsWechat) {
        return this.updateById(newsWechat);
    }

    @Transactional(rollbackFor={ Exception.class, BaseRunException.class },timeout = 120)
    @Override
    public boolean insertNewsAndContent(NewsWechat newsWechat, NewsWechatContent newsWechatContent, ProcessorParameter processorParameter) {
        // 参数校验
        if(StringUtils.isBlank(newsWechatContent.getContent())){
            throw new BaseRunException("[save data]news wechat content is null,the wechat pm is " + newsWechat.getWechatMp()
                    + " title is "+newsWechat.getTitle());
        }

        NewsWechat newsWechatOld = findByPmTitle(newsWechat.getWechatMp(), newsWechat.getTitle());
        newsWechat.setModifyTime(new Date());

        if(newsWechatOld == null){
            // 新增-获取id
            long id = idHelps.getId(IdKeyConstant.FIELD_WECHAT);

            newsWechat.setId(id);
            this.insertWechat(newsWechat);

            newsWechatContent.setId(id);
            newsWechatContentService.insert(newsWechatContent);

            logger.info(MonitorConstant.MONITOR, "微信新闻数据插入成功 {}", MonitorHelper.ofSuccessLog(MonitorConstant.SAVE_INSERT, processorParameter));
        }else{
            // 修改
            newsWechat.setId(newsWechatOld.getId());
            if(WechatConstant.STATUS_RENEWAL == newsWechatOld.getStatus().intValue()){
                newsWechat.setStatus(WechatConstant.STATUS_VALID);
            } else {
                newsWechat.setStatus(newsWechatOld.getStatus());
            }
            this.updateWechatById(newsWechat);

            newsWechatContent.setId(newsWechatOld.getId());
            newsWechatContentService.updateById(newsWechatContent);

            logger.info(MonitorConstant.MONITOR, "微信新闻数据更新成功 {}", MonitorHelper.ofSuccessLog(MonitorConstant.SAVE_UPDATE, processorParameter));
        }
        return true;
    }

    @Override
    public List<Map<String,String>> findNewsByModifyTime(int amount) {
        Wrapper<NewsWechat> wrapperWeChat = new EntityWrapper<NewsWechat>();
        wrapperWeChat.ge("modify_time", DateUtils.addField(new Date(), Calendar.DAY_OF_YEAR,amount));
        List<NewsWechat> weChatNews = this.selectList(wrapperWeChat);
        if(CollectionUtils.isEmpty(weChatNews)){
            return null;
        }
        List<Map<String,String>> result = new ArrayList<>();
        for(NewsWechat newsWechat : weChatNews){
            Map<String,String> map = new HashMap<>();
            NewsWechatContent content = newsWechatContentService.selectById(newsWechat.getId());
            map.put("url",newsWechat.getWechatMp()+"->"+newsWechat.getRegion());
            map.put("title",newsWechat.getTitle());
            map.put("content",content.getContent());
            map.put("modify_time", DateUtils.convertDateToString(newsWechat.getModifyTime(),"yyyyMMdd"));
            result.add(map);
        }
        return result;
    }

    @Override
    public Map<String, String> findNewsById(Long id) {
        Map<String, String> result = new HashMap<>();
        NewsWechat newsWechat = this.selectById(id);
        NewsWechatContent content = newsWechatContentService.selectById(id);
        if(null != newsWechat && null != content){
            result.put("url",newsWechat.getWechatMp()+"->"+newsWechat.getRegion());
            result.put("title",newsWechat.getTitle());
            result.put("content",content.getContent());
            result.put("modify_time", DateUtils.convertDateToString(newsWechat.getModifyTime(),"yyyyMMdd"));
        }
        return result;
    }

    @Override
    public List<ContentVo> findList(ContentDto contentDto) {
        return newsWechatMapper.findList(contentDto);
    }
}
