package com.bxm.spider.deal.dal.service.impl;

import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.bxm.spider.cache.RedisClient;
import com.bxm.spider.cache.constant.IdKeyConstant;
import com.bxm.spider.constant.exception.BaseRunException;
import com.bxm.spider.constant.monitor.MonitorConstant;
import com.bxm.spider.constant.monitor.MonitorHelper;
import com.bxm.spider.constant.processor.ProcessorParameter;
import com.bxm.spider.deal.common.constant.ContentConstant;
import com.bxm.spider.deal.common.utils.StringHelp;
import com.bxm.spider.deal.dal.mapper.WechatNewsMapper;
import com.bxm.spider.deal.dal.service.WechatNewsContentDbService;
import com.bxm.spider.deal.dal.service.WechatNewsDbService;
import com.bxm.spider.deal.dal.utils.MultiPageHelps;
import com.bxm.spider.deal.facade.model.ContentDto;
import com.bxm.spider.deal.facade.model.ContentVo;
import com.bxm.spider.deal.model.dao.WechatNews;
import com.bxm.spider.deal.model.dao.WechatNewsContent;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;

/**
 * <p>
 * 爬取微信新闻表(分表) 服务实现类
 * </p>
 *
 * @author stephen
 * @since 2019-01-03
 */
@Service
public class WechatNewsDbServiceImpl extends ServiceImpl<WechatNewsMapper, WechatNews> implements WechatNewsDbService {
    private static final Logger logger = LoggerFactory.getLogger(WechatNewsDbServiceImpl.class);

    @Autowired
    private WechatNewsMapper wechatNewsMapper;

    @Autowired
    private WechatNewsContentDbService wechatNewsContentService;

    @Autowired
    private RedisClient redisClient;

    @Override
    public int insertNews(WechatNews wechatNews) {
        wechatNews.setSuffix(StringHelp.getHashCodeModTen(wechatNews.getTitle()));
        Long id = redisClient.hincrByOne(IdKeyConstant.getPrimaryId(), IdKeyConstant.FIELD_WECHATNEWS);
        if (null == id) {
            id = StringHelp.getUUIDHashCode(14);
        }
        wechatNews.setId(id * 10 + Long.parseLong(wechatNews.getSuffix()));
        return wechatNewsMapper.insertReturnId(wechatNews);
    }

    @Override
    public int updateNewsById(WechatNews news) {
        if (StringUtils.isBlank(news.getSuffix())) {
            news.setSuffix(StringHelp.getHashCodeModTen(news.getTitle()));
        }
        return wechatNewsMapper.updateNewsById(news);
    }

    @Transactional(rollbackFor = {Exception.class, BaseRunException.class}, timeout = 120)
    @Override
    public void insertNewsAndContent(WechatNews news, WechatNewsContent newsContent, ProcessorParameter processorParameter) {
        if (StringUtils.isBlank(newsContent.getContent())) {
            throw new BaseRunException("保存微信新闻内容为空，title: " + news.getTitle());
        }
        WechatNews newsOld = findNewsByTitle(news.getTitle());
        news.setTitleCode(news.getTitle().hashCode());
        if (null == newsOld) {
            insertNews(news);
            if (null == news.getId()) {
                throw new BaseRunException("保存微信新闻返回id为空，title: " + news.getTitle());
            }
            newsContent.setId(news.getId());
            wechatNewsContentService.insert(news.getTitle(), newsContent);
            logger.info(MonitorConstant.MONITOR, "微信新闻数据插入成功 {}", MonitorHelper.ofSuccessLog(MonitorConstant.SAVE_INSERT, processorParameter));
        } else if (newsOld.getStatus() != null && ContentConstant.STATUS_RENEWAL == newsOld.getStatus().intValue()) {
            news.setId(newsOld.getId());
            news.setStatus(ContentConstant.STATUS_VALID);
            updateNewsById(news);
            newsContent.setId(newsOld.getId());
            wechatNewsContentService.updateById(news.getTitle(), newsContent);
            logger.info(MonitorConstant.MONITOR, "微信新闻数据更新成功 {}", MonitorHelper.ofSuccessLog(MonitorConstant.SAVE_UPDATE, processorParameter));
        }
    }

    @Override
    public WechatNews findNewsByTitle(String title) {
        return wechatNewsMapper.findNewsByTitle(StringHelp.getHashCodeModTen(title), title.hashCode());
    }

    @Override
    public Map<String, String> findNewsById(Long id, String suffix) {
        return wechatNewsMapper.findNewsById(id, suffix);
    }

    @Override
    public List<ContentVo> listContent(ContentDto contentDto) {
        return MultiPageHelps.list(contentDto, wechatNewsMapper);
    }

}
