package com.bxm.spider.download.integration.impl;

import com.bxm.spider.constant.monitor.*;
import com.bxm.spider.constant.task.TaskConstant;
import com.bxm.spider.deal.facade.model.DealDto;
import com.bxm.spider.deal.facade.service.ProcessorFacadeService;
import com.bxm.spider.download.facade.model.DownloadDto;
import com.bxm.spider.download.integration.DealIntegrationService;
import com.bxm.warcar.utils.response.ResponseModel;
import com.bxm.warcar.utils.response.ResponseModelFactory;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Objects;

/**
 * 调用处理中心接口服务
 *
 * @ClassName DealIntegrationServiceImpl
 * @CopyRright (c) 2018-bxm：杭州微财网络科技有限公司
 * @Author zhaox
 * @Date 2018/10/23 10:25
 * @Version 1.0.0
 * @Modifier zhaox
 * @Modify Date 2018/10/23 10:25
 **/
@Service
public class DealIntegrationServiceImpl implements DealIntegrationService {

    @Autowired
    ProcessorFacadeService processorFacadeService;

    private static final Logger LOG = LoggerFactory.getLogger(DealIntegrationServiceImpl.class);

    /**
     * 调处理中心
     *
     * @param downloadDto
     * @param content
     * @return
     */
    @Override
    public MonitorDto dealContent(DownloadDto downloadDto, String content) {
        ResponseModel<Boolean> responseModel = ResponseModelFactory.FAILED();

        // 参数转换
        DealDto dealDto = convertToDealDto(downloadDto, content, null);

        // 调用处理中心
        try {
            responseModel = processorFacadeService.processor(dealDto);
            LOG.info("【调用处理中心成功】,response data is {}",responseModel.getData());
        } catch (Exception e) {
            LOG.error("【调用处理中心失败】！{}", downloadDto, e);

            // 请求失败自动重试
            for (int i = 0; i < TaskConstant.RETRY_TIME; i++) {
                try {
                    responseModel = processorFacadeService.processor(dealDto);
                    if (responseModel.isSuccess()) {
                        break;
                    }
                } catch (Exception e1) {
                    LOG.error("重试调用处理中心失败{}次！data: {}, message: {}", i, dealDto, e1.getMessage());
                }
            }
            if(responseModel.isError()){
                LOG.error(MonitorConstant.MONITOR, "重试调用处理中心失败！data: {}, {}", dealDto, MonitorHelper.ofLog(false,
                        MonitorConstant.DOWNLOAD_PROGRESS, downloadDto.getProcessorParameter(), ErrorEnum.DOWNLOAD_DEAL_ERROR, e.getMessage()), e);
                return MonitorDtoHelper.ofMonitorDtoFail(dealDto.getProcessorParameter(), e.getMessage());
            }
        }
        return MonitorDtoHelper.ofMonitorDtoSuccess(dealDto.getProcessorParameter());
    }

    @Override
    public MonitorDto dealImage(DownloadDto downloadDto, String imageStr) {
        ResponseModel<Boolean> responseModel = ResponseModelFactory.FAILED();
        DealDto dealDto = convertToDealDto(downloadDto, null, imageStr);
        try {
            responseModel = processorFacadeService.imageProcessor(dealDto);
            LOG.info("【图片下载调用处理中心成功】,response data is {}",responseModel.getData());
        } catch (Exception e) {
            LOG.error("【图片下载调用处理中心失败】{}", dealDto, e);

            // 请求失败自动重试
            for (int i = 0; i < TaskConstant.RETRY_TIME; i++) {
                try {
                    responseModel = processorFacadeService.imageProcessor(dealDto);
                    if (responseModel.isSuccess()) {
                        break;
                    }
                } catch (Exception e1) {
                    LOG.error("重试调用处理中心失败{}次！data: {}, message: {}", i, dealDto, e1.getMessage());
                }
            }

            if(responseModel.isError()){
                LOG.error("重试调用处理中心失败！data: {}", dealDto, e);
                return MonitorDtoHelper.ofMonitorDtoFail(dealDto.getProcessorParameter(), e.getMessage());
            }
        }
        return MonitorDtoHelper.ofMonitorDtoSuccess(dealDto.getProcessorParameter());
    }

    /**
     * 转换为处理中心调用参数对象
     * @param downloadDto
     * @return
     */
    private DealDto convertToDealDto(DownloadDto downloadDto, String pageInfo, String imageStr) {
        DealDto dealDto = new DealDto();
        if (StringUtils.isNotBlank(imageStr)){
            dealDto.setImageStr(imageStr);
        }
        if (StringUtils.isNotBlank(pageInfo)){
            dealDto.setPageInfo(pageInfo.replaceAll("\n|\t|\r"," "));
        }
        if (!Objects.isNull(downloadDto)){
            dealDto.setProcessorParameter(downloadDto.getProcessorParameter());
        }
        return dealDto;
    }
}
