package com.bxm.spider.monitor.service.service.impl;

import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.bxm.spider.constant.monitor.MonitorConstant;
import com.bxm.spider.constant.monitor.MonitorLogDto;
import com.bxm.spider.monitor.dal.mapper.SpiderExceptionMonitorMapper;
import com.bxm.spider.monitor.dal.mapper.SpiderMonitorReportMapper;
import com.bxm.spider.monitor.model.bo.SpiderMonitorReportBo;
import com.bxm.spider.monitor.model.dao.SpiderExceptionMonitor;
import com.bxm.spider.monitor.model.dao.SpiderMonitorReport;
import com.bxm.spider.monitor.service.service.MonitorLogService;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 监控日志服务实现
 *
 * @ClassName MonitorLogServiceImpl
 * @CopyRright (c) 2018-bxm：杭州微财网络科技有限公司
 * @Author stephen
 * @Date 2018/12/19 15:15
 * @Version 1.0
 **/
@Service
public class MonitorLogServiceImpl implements MonitorLogService {
    private static final Logger logger = LoggerFactory.getLogger(MonitorLogServiceImpl.class);
    public static final String MONITOR_TIME_FORMAT = "yyyy-MM-dd HH:00:00";

    @Autowired
    private SpiderMonitorReportMapper spiderMonitorReportMapper;

    @Autowired
    private SpiderExceptionMonitorMapper spiderExceptionMonitorMapper;

    /**
     * 主map  key:hashCode->MonitorUtil.getHashCode  value:SpiderMonitorReportBo
     */
    private ConcurrentHashMap<Integer, SpiderMonitorReportBo> mainMap = new ConcurrentHashMap<>();

    /**
     * 备用map  key:hashCode->MonitorUtil.getHashCode  value:SpiderMonitorReportBo
     */
    private ConcurrentHashMap<Integer, SpiderMonitorReportBo> minorMap = new ConcurrentHashMap<>();

    /**
     * 是否使用主map
     */
    private boolean isMain = true;

    private ThreadPoolExecutor threadPoolExecutor;

    public MonitorLogServiceImpl() {
        this.threadPoolExecutor = new ThreadPoolExecutor(5, 5,
                0, TimeUnit.SECONDS, new LinkedBlockingQueue<>(), new NamedThreadFactory());
    }

    @Override
    public boolean saveResult(MonitorLogDto monitorLogDto) {
        threadPoolExecutor.execute(() -> {
            if (monitorLogDto.getSuccess()) {
                try {
                    SpiderMonitorReportBo spiderMonitorReportBo = getFromMap(monitorLogDto.getSerialNum(), monitorLogDto.getUrlType());
                    doIncrement(monitorLogDto.getMonitorFlow(), spiderMonitorReportBo);
                } catch (Exception e) {
                    logger.error("save monitor report error, data: {}", monitorLogDto, e);
                }
            } else {
                try {
                    SpiderExceptionMonitor spiderExceptionMonitor = new SpiderExceptionMonitor();
                    spiderExceptionMonitor.setSerialNum(monitorLogDto.getSerialNum());
                    spiderExceptionMonitor.setUrlType(monitorLogDto.getUrlType());
                    spiderExceptionMonitor.setFlowCode(monitorLogDto.getMonitorFlow());
                    spiderExceptionMonitor.setErrorCode(monitorLogDto.getErrorCode());
                    spiderExceptionMonitor.setErrorMessage(monitorLogDto.getErrorMessage());

                    spiderExceptionMonitorMapper.insert(spiderExceptionMonitor);
                    logger.info("save exception data success, data: {}", monitorLogDto);
                } catch (Exception e) {
                    logger.error("save exception data error, data: {}", monitorLogDto, e);
                }
            }
        });
        return true;
    }

    @Scheduled(cron = "0 */5 * * * ?")
    private void scheduledSaveDB() {
        logger.info("save monitor report to database start, mainMap size: {}, minorMap size：{}, isMain: {}", mainMap.size(), minorMap.size(), isMain);
        // 切换内存的主备
        isMain = !isMain;
        if (isMain) {
            saveDB(minorMap);
        } else {
            saveDB(mainMap);
        }
    }

    private SpiderMonitorReportBo getFromMap(String serialNum, String urlType) {
        Integer key = getHashCode(serialNum, urlType);
        SpiderMonitorReportBo spiderMonitorReportBo = isMain ? mainMap.get(key) : minorMap.get(key);

        if (null == spiderMonitorReportBo) {
            spiderMonitorReportBo = new SpiderMonitorReportBo();
            spiderMonitorReportBo.setSerialNum(serialNum);
            spiderMonitorReportBo.setUrlType(urlType);
            spiderMonitorReportBo.setMonitorTime(DateHelper.format(new Date(), MONITOR_TIME_FORMAT));
        }

        if (isMain) {
            mainMap.put(key, spiderMonitorReportBo);
        } else {
            minorMap.put(key, spiderMonitorReportBo);
        }

        return spiderMonitorReportBo;
    }

    private void saveDB(ConcurrentHashMap<Integer, SpiderMonitorReportBo> dataMap) {
        if (dataMap.isEmpty()) {
            return;
        }
        try {
            for (Map.Entry<Integer, SpiderMonitorReportBo> entry : dataMap.entrySet()) {
                SpiderMonitorReport spiderMonitorReport = ofSpiderMonitorReport(entry.getValue());
                List<SpiderMonitorReport> list = spiderMonitorReportMapper.selectList(new EntityWrapper()
                        .eq("serial_num", spiderMonitorReport.getSerialNum())
                        .eq("url_type", spiderMonitorReport.getUrlType())
                        .eq("monitor_time", spiderMonitorReport.getMonitorTime()));

                if (CollectionUtils.isEmpty(list)) {
                    spiderMonitorReportMapper.insert(spiderMonitorReport);
                } else {
                    SpiderMonitorReport report = list.get(0);
                    updateSpiderMonitorReport(report, spiderMonitorReport);
                    spiderMonitorReportMapper.updateById(report);
                }
            }

            logger.info("spider_monitor_report save success, map.size: {}", isMain == true ? minorMap.size() : mainMap.size());
        } catch (Exception e) {
            logger.error("spider_monitor_report save error, mainMap: {}, minorMap: {}", mainMap, minorMap, e);
        }

        // 保存结束清空
        dataMap.clear();
    }

    private SpiderMonitorReport ofSpiderMonitorReport(SpiderMonitorReportBo spiderMonitorReportBo) {
        SpiderMonitorReport spiderMonitorReport = new SpiderMonitorReport();

        spiderMonitorReport.setSerialNum(spiderMonitorReportBo.getSerialNum());
        spiderMonitorReport.setUrlType(spiderMonitorReportBo.getUrlType());
        spiderMonitorReport.setMonitorTime(DateHelper.parse(spiderMonitorReportBo.getMonitorTime(), MONITOR_TIME_FORMAT));
        spiderMonitorReport.setProdStart(spiderMonitorReportBo.getProdStart().intValue());
        spiderMonitorReport.setProdEnd(spiderMonitorReportBo.getProdEnd().intValue());
        spiderMonitorReport.setDownloadStart(spiderMonitorReportBo.getDownloadStart().intValue());
        spiderMonitorReport.setDownloadEnd(spiderMonitorReportBo.getDownloadEnd().intValue());
        spiderMonitorReport.setDealStart(spiderMonitorReportBo.getDealStart().intValue());
        spiderMonitorReport.setDealEnd(spiderMonitorReportBo.getDealEnd().intValue());
        spiderMonitorReport.setSaveStart(spiderMonitorReportBo.getSaveStart().intValue());
        spiderMonitorReport.setSaveEnd(spiderMonitorReportBo.getSaveEnd().intValue());

        return spiderMonitorReport;
    }

    private void updateSpiderMonitorReport(SpiderMonitorReport existedReport, SpiderMonitorReport newReport) {
        existedReport.setProdStart(existedReport.getProdStart() + newReport.getProdStart());
        existedReport.setProdEnd(existedReport.getProdEnd() + newReport.getProdEnd());
        existedReport.setDownloadStart(existedReport.getDownloadStart() + newReport.getDownloadStart());
        existedReport.setDownloadEnd(existedReport.getDownloadEnd() + newReport.getDownloadEnd());
        existedReport.setDealStart(existedReport.getDealStart() + newReport.getDealStart());
        existedReport.setDealEnd(existedReport.getDealEnd() + newReport.getDealEnd());
        existedReport.setSaveStart(existedReport.getSaveStart() + newReport.getSaveStart());
        existedReport.setSaveEnd(existedReport.getSaveEnd() + newReport.getSaveEnd());
    }

    private void doIncrement(String monitorFlow, SpiderMonitorReportBo spiderMonitorReportBo) {
        if (MonitorConstant.PROD_START.equals(monitorFlow)) {
            spiderMonitorReportBo.getProdStart().getAndIncrement();
        } else if (MonitorConstant.PROD_END.equals(monitorFlow)) {
            spiderMonitorReportBo.getProdEnd().getAndIncrement();
        } else if (MonitorConstant.DOWNLOAD_START.equals(monitorFlow)) {
            spiderMonitorReportBo.getDownloadStart().getAndIncrement();
        } else if (MonitorConstant.DOWNLOAD_END.equals(monitorFlow)) {
            spiderMonitorReportBo.getDownloadEnd().getAndIncrement();
        } else if (MonitorConstant.DEAL_START.equals(monitorFlow)) {
            spiderMonitorReportBo.getDealStart().getAndIncrement();
        } else if (MonitorConstant.DEAL_END.equals(monitorFlow)) {
            spiderMonitorReportBo.getDealEnd().getAndIncrement();
        } else if (MonitorConstant.SAVE_START.equals(monitorFlow)) {
            spiderMonitorReportBo.getSaveStart().getAndIncrement();
        } else if (MonitorConstant.SAVE_END.equals(monitorFlow)) {
            spiderMonitorReportBo.getSaveEnd().getAndIncrement();
        }

        logger.debug("doIncrement, monitorFlow: {}, spiderMonitorReportBo: {}", monitorFlow, spiderMonitorReportBo);
    }

    private Integer getHashCode(String serialNum, String urlType) {
        String monitorTime = DateHelper.format(new Date(), MONITOR_TIME_FORMAT);
        String str = serialNum + urlType + monitorTime;
        return str.hashCode();
    }
}
