package com.bxm.warcar.mq.autoconfigure;

import com.bxm.warcar.mq.Consumer;
import com.bxm.warcar.mq.Listener;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;

import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 消费者构造工厂。
 *
 * <p>自动查找所有注册 {@link Listener} 的消息消费监听器，并为这些监听器创建消费者。</p>
 *
 * @author allen
 * @date 2021-09-02
 * @since 1.0
 */
public abstract class AbstractConsumerBeanFactory implements DisposableBean, ApplicationListener<ApplicationReadyEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractConsumerBeanFactory.class);

    private final ConcurrentHashMap<String, Consumer> consumers = new ConcurrentHashMap<>();

    protected abstract Consumer create(String consumerId, Listener messageListener);

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        ConfigurableApplicationContext applicationContext = event.getApplicationContext();
        Map<String, Listener> beans = applicationContext.getBeansOfType(Listener.class);
        if (MapUtils.isEmpty(beans)) {
            return;
        }
        for (Map.Entry<String, Listener> entry : beans.entrySet()) {
            Listener listener = entry.getValue();
            register(listener);
        }
    }

    @Override
    public void destroy() {
        for (Map.Entry<String, Consumer> entry : consumers.entrySet()) {
            String key = entry.getKey();
            Consumer consumer = entry.getValue();

            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Consumer {} on shutdown", key);
            }

            consumer.shutdown();
        }
    }

    private void register(Listener listener) {
        String consumerId = listener.getConsumerId();
        if (StringUtils.isBlank(consumerId)) {
            LOGGER.warn("{}'s consumerId value is null!", listener);
            return;
        }

        Consumer consumer = create(consumerId, listener);
        if (null == consumer) {
            return;
        }

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("Registering consumer: {} of topic: {}", consumerId, listener.getTopic());
        }
        consumer.start();

        consumers.put(consumerId, consumer);
    }

    public Map<String, Consumer> getConsumers() {
        return Collections.unmodifiableMap(consumers);
    }
}
