package com.bxm.warcar.web.interceptor;

import com.bxm.warcar.cache.impls.redis.JedisFetcher;
import com.bxm.warcar.utils.IpHelper;
import com.bxm.warcar.web.autoconfigure.WebProperties;
import com.bxm.warcar.web.bo.PositionMonitorBo;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * <h3>Ip拦截器  </h3>
 * <p>
 * 拦截后返回结果是个空对象
 *
 * 修改 by kk 2019-11-25 17:32:17
 * 逻辑变更：作弊IP，有广告位配置策略则统计埋点数据，否则不进行统计
 *
 * @author hcmony
 * @see com.bxm.warcar.utils.response.ResponseModel
 * </p>
 * @since V1.0.0, 2019/03/07 10:18
 */
public class IpInterceptor implements HandlerInterceptor {

	private static final Logger LOGGER = LoggerFactory.getLogger(IpInterceptor.class);
	private static final String RESPONSE_MODEL = "{\"data\":null,\"success\":true,\"error\":false,\"code\":1,\"message\":\"ip\"}";
	/**示例：2BB2F4CAD877442JUKQL23EC2D9B1Y8KO*/
	private static final String UID = "uid";
	/**示例：2bb2f4cad877442187023ec2d9b936f6*/
	private static final String APPKEY = "appkey";
	/**示例：money-1*/
	private static final String BUSINESS = "business";

	private final WebProperties webProperties;
	private final JedisFetcher sentinelJedisFetcher;
	private final JedisFetcher visionJedisFetcher;

	public IpInterceptor(JedisFetcher sentinelJedisFetcher, JedisFetcher visionJedisFetcher, WebProperties webProperties) {
		this.webProperties = webProperties;
		this.sentinelJedisFetcher = sentinelJedisFetcher;
		this.visionJedisFetcher = visionJedisFetcher;
	}

	@Override
	public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
		try {
			// 作弊流量过滤开关是否开启, 关闭则直接放行
			if(!isOpenFilter()){
				return true;
			}

			// 这段逻辑保留，忽略某些具有特殊参数的请求如包含某些model_type，或者特异uid等参数
			// 当前没有用，但保留一个入口
			if (ignore(request)) {
				return true;
			}

			String ip = StringUtils.defaultIfEmpty(IpHelper.getIpFromHeader(request), "");

			String uid = StringUtils.defaultIfEmpty(getValue(request, UID), "");

			boolean cheatIp = checkIpCheat(ip);
			boolean cheatUid = checkUidCheat(uid);
			// IP,UID均未作弊则放行
			if(!cheatIp && !cheatUid){
				return true;
			}

			String appkey = StringUtils.defaultIfEmpty(getValue(request, APPKEY), "");
			String business = StringUtils.defaultIfEmpty(getValue(request, BUSINESS), "");

			boolean monitorPosition = checkMonitorPosition(appkey, business, cheatIp, cheatUid);
			if(monitorPosition){
				return true;
			}else{
				write(response);
				return false;
			}
		} catch (Exception e) {
			LOGGER.error(" Resolving IP exceptions !", e);
		}
		return true;
	}


	private boolean ignore(HttpServletRequest request) {
		WebProperties.Ip ip = webProperties.getIp();
		if (null == ip) {
			return false;
		}
		Map<String, Set<String>> ignoreParameters = ip.getIgnoreParameters();
		if (MapUtils.isNotEmpty(ignoreParameters)) {
			for (Map.Entry<String, Set<String>> entry : ignoreParameters.entrySet()) {
				String key = entry.getKey();
				String pvalue = request.getParameter(key);
				if (entry.getValue().contains(pvalue)) {
					return true;
				}
			}
		}
		return false;
	}

	private boolean checkMonitorPosition(String appkey, String business, boolean cheatIp, boolean cheatUid) {
		String position = appkey + (business.replaceAll("money|ad", ""));
		PositionMonitorBo positionMonitorBo = sentinelJedisFetcher.hfetch(() -> "AD:CHEAT:MONITOR:POSITION", position, PositionMonitorBo.class);
		if(Objects.nonNull(positionMonitorBo)){
			Set<String> monitorGranularitySet = (Set<String>) positionMonitorBo.getSet(PositionMonitorBo.MonitorEnum.MONITOR_GRANULARITY);
			if(monitorGranularitySet.contains(PositionMonitorBo.IP) && cheatIp){
				return true;
			}
			if(monitorGranularitySet.contains(PositionMonitorBo.UID) && cheatUid){
				return true;
			}
		}

		PositionMonitorBo all = sentinelJedisFetcher.hfetch(() -> "AD:CHEAT:MONITOR:POSITION", "ALL", PositionMonitorBo.class);
		if(Objects.nonNull(all)){
			Set<PositionMonitorBo.MonitorEnum> processMode = all.getProcessMode();
			// 特殊redis key 处理
			if(CollectionUtils.isEmpty(processMode)){
				return false;
			}else if(processMode.size() == 1 && processMode.contains(PositionMonitorBo.MonitorEnum.MONITOR_GRANULARITY)){
				return false;
			}else{
				Set<String> monitorGranularitySet = (Set<String>) all.getSet(PositionMonitorBo.MonitorEnum.MONITOR_GRANULARITY);
				if(monitorGranularitySet.contains(PositionMonitorBo.IP) && cheatIp){
					return true;
				}
				if(monitorGranularitySet.contains(PositionMonitorBo.UID) && cheatUid){
					return true;
				}
			}
		}
		return false;
	}

	private boolean checkUidCheat(String uid) {
		Object o = visionJedisFetcher.fetch(() -> StringUtils.join(new String[]{"CHEAT", "UID", uid}, ":"), Object.class);
		return Objects.nonNull(o);
	}

	private boolean checkIpCheat(String ip) {
		Object o = visionJedisFetcher.fetch(() -> StringUtils.join(new String[]{"CHEAT", "IP", ip}, ":"), Object.class);
		return Objects.nonNull(o);
	}

	private String getValue(HttpServletRequest request, String key){
		String value = request.getParameter(key);
		return value;
	}

	/**
	 * 作弊流量过滤开关是否开启
	 *
	 * @param
	 * @return boolean
	 * @throws
	 * @author kk.xie
	 * @date 2019/11/25 17:53
	 */
	private boolean isOpenFilter(){
		Boolean isOpenFilter = sentinelJedisFetcher.fetch(() -> "AD:CHEAT:FLOW:FILTER:OPTION", Boolean.class);
		if(Boolean.TRUE.equals(isOpenFilter)){
			return true;
		}
		return false;
	}

	private void write(HttpServletResponse response) {
		PrintWriter writer = null;
		try {
			writer = response.getWriter();
			response.setStatus(HttpServletResponse.SC_OK);
			writer.print(RESPONSE_MODEL);
			writer.flush();
		} catch (IOException e) {
			LOGGER.error(" Resolving IP write !", e);
		} finally {
			IOUtils.closeQuietly(writer);
		}
	}


	@Override
	public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {

	}


	@Override
	public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {

	}
}
