package com.bxm.warcar.integration.autoconfigure.message;

import com.bxm.warcar.integration.message.MessageProducer;
import com.bxm.warcar.integration.message.annotation.Messaging;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.Configuration;

import javax.annotation.PreDestroy;
import java.lang.reflect.Method;

/**
 * <h3>消息切面，实现了后置通知。</h3>
 * <p>当被拦截的方法执行完成后，会将请求参数和返回对象构造成一个{@link com.bxm.warcar.integration.message.MessageBody}对象，然后对其序列化后发送到消息队列。</p>
 *
 * @see com.bxm.warcar.integration.message.MessageBody
 * @see Messaging
 *
 * @author allen
 * @since V1.0.0 2017/12/08
 */
@Aspect
@Configuration
@ConditionalOnBean(MessageProducer.class)
public class MessageAspect {

    private static final Logger LOGGER = LoggerFactory.getLogger(MessageAspect.class);

    private final MessageProducer messageProducer;

    public MessageAspect(MessageProducer messageProducer) {
        this.messageProducer = messageProducer;
    }

    @PreDestroy
    public void destroy() {
        this.messageProducer.getThreadPool().shutdown();
    }

    @AfterReturning(pointcut = "@annotation(com.bxm.warcar.integration.message.annotation.Messaging)", returning = "returning")
    public void doAfterReturning(JoinPoint point, Object returning) {
        Object[] args = point.getArgs();

        Method method = getMethod(point);
        if (null == method) {
            return;
        }
        if (null == returning) {
            return;
        }

        Messaging annotation = method.getAnnotation(Messaging.class);

        this.messageProducer.asyncSendMessage(args, returning, annotation);
    }

    private Method getMethod(JoinPoint point) {
        MethodSignature methodSignature = (MethodSignature) point.getSignature();

        Class<?> targetClass = point.getTarget().getClass();
        try {
            return targetClass.getMethod(methodSignature.getName(), methodSignature.getParameterTypes());
        } catch (NoSuchMethodException e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("getMethod:", e);
            }
            return null;
        }
    }

    public MessageProducer getMessageProducer() {
        return messageProducer;
    }

}
