package com.bxm.warcar.algorithm.flow.filter;

import com.bxm.warcar.algorithm.config.NumericalConfiguration;
import com.bxm.warcar.algorithm.NumericalModel;
import com.bxm.warcar.algorithm.RequestModel;
import com.bxm.warcar.algorithm.utils.DoubleUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * <h3>测试数据与正式数据  流量分配</h3>
 * <p></p>
 *
 * @author hcmony
 * @since V1.0.0, 2019/04/17 13:31
 */
@Component
public class DistributionFilter extends AbstractFilter {

	private static final double DEFAULT_RATE = 1.0;

	@Autowired
	private NumericalConfiguration numericalConfiguration;

	@Override
	protected void doFilter(RequestModel requestModel) {
		List<NumericalModel> testList = requestModel.getTestList();
		List<NumericalModel> formalList = requestModel.getFormalList();

		/**
		 * 测试数据流量分配规则：
		 * 		单个素材的流量分配：设广告位前一日的曝光PV为x,素材的测试阈值（等于三级阈值）为y，y/x*100%即为这个活动在这个广告位的随机流量比例。
		 * 		测试阶段总流量切分：广告位日常用来测试的流量不超过总流量的20%（除非这个广告位没有一个正式素材，就100%测试）
		 * 正式数据流量分配规则：
		 * 		如果不是队列的最后一个 nst=正式阶段总流量*（1-x)^（n-1)*x%
		 *		如果是队列的最后一个 nst=正式阶段总流量*(1-x)%^（n-1)*100%
		 *		x=0.8 数据库可配置
		 */

		//如果正式数据为空，那么测试数据流量为100%
		if (CollectionUtils.isEmpty(formalList)){

			if (CollectionUtils.isEmpty(testList)){
				return;
			}
			doTestFlow(testList,DEFAULT_RATE);
			return;
		}

		//如果测试数据为空，那么正式数据流量为100%
		if (CollectionUtils.isEmpty(testList)){
			if (CollectionUtils.isEmpty(formalList)){
				return;
			}
			doFormalFlow( formalList,DEFAULT_RATE);
			return;
		}

		//如果测试与正式都有数据，那么按照比例分配测试流量
		double allTestFlowRate = doTestFlow(testList, numericalConfiguration.getTestFlow());

		//如果测试与正式都有数据，那么按照比例分配正式流量
		double formalFlow = DoubleUtils.subtract(DEFAULT_RATE,allTestFlowRate);
		doFormalFlow( formalList,formalFlow);
	}


	private double doTestFlow(List<NumericalModel> testList, double rate){
		double allFlowRate = 0.0;

		NumericalModel numericalModel = testList.get(0);
		testFlow(numericalModel,rate);

		// 按设定测试流量正常分配，达到了剩下的给0
		for (int i=0;i<testList.size();i++){
			NumericalModel model = testList.get(i);
			model.setFlowRate(numericalModel.getFlowRate());
			allFlowRate += model.getFlowRate();
			//流量比总和不能超过分配的流量
			if (allFlowRate+model.getFlowRate()>rate ){
				return allFlowRate;
			}
		}
		if (rate<DEFAULT_RATE){
			return allFlowRate;
		}

		// 如果测试流量加起来没有设定测试流量多的时候，得按 测试流量均分
		if (allFlowRate+numericalModel.getFlowRate()<rate){
			double averageFlowRate = DoubleUtils.divide(rate,testList.size());
			for(int i=0;i<testList.size();i++){
				NumericalModel model = testList.get(i);
				model.setFlowRate(averageFlowRate);
			}
			allFlowRate =  rate;
		}
		return allFlowRate;
	}


	/**
	 * 设广告位前一日的曝光PV为x,素材的测试阈值（等于三级阈值）为y，y/x*100%即为这个活动在这个广告位的随机流量比例
	 * @param model
	 * @param rate
	 */
	private void testFlow(NumericalModel model,double rate){
		long preDatePv = model.getPreGroupDatePv();
		if (preDatePv< numericalConfiguration.getDefaultOpenPv()){
			preDatePv = numericalConfiguration.getDefaultOpenPv();
		}
		double flowRate = DoubleUtils.divide(numericalConfiguration.getThreshold3(),preDatePv);
		model.setFlowRate(flowRate);
	}

	private void doFormalFlow(List<NumericalModel> formalList,double rate){
		if (formalList.size()==1){
			NumericalModel model = formalList.get(0);
			model.setFlowRate(rate);
			return;
		}

		for (int i=0;i<formalList.size();i++){
			NumericalModel model = formalList.get(i);
			if (i==formalList.size()-1){
				//nst=正式阶段总流量*(1-x)%^（n-1)*100%
				double personalFactor = numericalConfiguration.getPersonalFactor();
				double flowRate = DoubleUtils.multiply(rate,
						DoubleUtils.pow(DoubleUtils.subtract(1 ,personalFactor),i));
				model.setFlowRate(flowRate);
				continue;
			}
			formalFlow(model,rate,i);
		}
	}

	/**
	 * nst=正式阶段总流量*（1-x)^（n-1)*x%
	 * @param model
	 * @param rate
	 */
	private void formalFlow(NumericalModel model,double rate,int i){
		double personalFactor = numericalConfiguration.getPersonalFactor();
		double flowRate = DoubleUtils
				.multiply(rate,
						DoubleUtils.pow(DoubleUtils.subtract(1,personalFactor), i)
						,personalFactor);
		model.setFlowRate(flowRate);
	}


}
