/*
 * Copyright 2016 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * textile.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it only
 * in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */

package com.bxm.warcar.cache.impls.redis;

import com.bxm.warcar.cache.Windowed;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.base.Preconditions;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.ScanParams;
import redis.clients.jedis.ScanResult;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author allen
 * @since 1.0.0
 */
public class JedisWindowed implements Windowed {

    private static final Logger LOGGER = LoggerFactory.getLogger(JedisWindowed.class);

    private final JedisPool jedisPool;
    private final String primary;
    private final Duration windowLength;
    private final Duration slidingInterval;
    private final ScheduledExecutorService scheduled = new ScheduledThreadPoolExecutor(1,
            new NamedThreadFactory("sliding"));
    private volatile String currentWindowed;

    /**
     * @param jedisPool jedis pool
     * @param primary primary key
     * @param windowLength length of window
     * @param slidingInterval sliding interval
     */
    public JedisWindowed(JedisPool jedisPool, String primary, Duration windowLength, Duration slidingInterval) {
        Preconditions.checkNotNull(jedisPool);
        Preconditions.checkArgument(StringUtils.isNotBlank(primary));
        Preconditions.checkArgument(windowLength.value > 0 && windowLength.toMinutes() >= slidingInterval.toMinutes());
        Preconditions.checkArgument(slidingInterval.value > 0);

        this.jedisPool = jedisPool;
        this.primary = primary;
        this.windowLength = windowLength;
        this.slidingInterval = slidingInterval;

        this.startSlidingThread();
    }

    private void startSlidingThread() {
        scheduled.scheduleWithFixedDelay(this::sliding, 0, slidingInterval.value, slidingInterval.timeUnit);
    }

    @Override
    public double execute(double incr) {
        Jedis jedis = jedisPool.getResource();
        try {
            String field = this.currentWindowed;
            if (StringUtils.isBlank(field)) {
                this.refreshCurrentWindowed();
                field = this.currentWindowed;
            }
            return jedis.hincrByFloat(primary, field, incr);
        } catch (Exception e) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("execute: ", e);
            }
            return -1;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public double get() {
        Jedis jedis = jedisPool.getResource();
        try {
            double rst = 0;
            LocalDateTime purgeTime = getPurgeTime();
            Map<String, String> map = jedis.hgetAll(primary);
            Set<Map.Entry<String, String>> entries = map.entrySet();
            for (Map.Entry<String, String> entry : entries) {
                String k = entry.getKey();
                String val = entry.getValue();
                LocalDateTime ktime = convert2Time(k);
                if (!isExpiredWindow(purgeTime, ktime)) {
                    rst += NumberUtils.toDouble(val);
                }
            }
            return rst;
        } catch (Exception e) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("get: ", e);
            }
            return -1;
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    @Override
    public void destroy() {
        scheduled.shutdownNow();
    }

    private void sliding() {
        refreshCurrentWindowed();
        purgeWindow();
    }

    private void purgeWindow() {
        LocalDateTime purgeTime = getPurgeTime();
        Jedis jedis = jedisPool.getResource();
        try {
            String cursor = "0";
            boolean finished = false;
            Set<String> deleted = new HashSet<>();
            do {
                ScanResult<Map.Entry<String, String>> hscan = jedis.hscan(primary, cursor, new ScanParams().count(1));
                cursor = hscan.getStringCursor();
                List<Map.Entry<String, String>> result = hscan.getResult();
                if (CollectionUtils.isEmpty(result)) {
                    finished = true;
                } else {
                    for (Map.Entry<String, String> entry : result) {
                        String k = entry.getKey();
                        LocalDateTime ktime = convert2Time(k);
                        if (isExpiredWindow(purgeTime, ktime)) {
                            deleted.add(k);
                        } else {
                            finished = true;
                            break;
                        }
                    }
                }
            } while (!finished);
            if (CollectionUtils.isNotEmpty(deleted)) {
                jedis.hdel(primary, deleted.toArray(new String[0]));
            }
        } catch (Exception e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("purgeWindow: ", e);
            }
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }

    private void refreshCurrentWindowed() {
        this.currentWindowed = convert2Field(LocalDateTime.now());
    }

    private boolean isExpiredWindow(LocalDateTime purgeTime, LocalDateTime fieldTime) {
        return fieldTime.isBefore(purgeTime) || fieldTime.isEqual(purgeTime);
    }

    private LocalDateTime getPurgeTime() {
        return LocalDateTime.now()
                    .minusMinutes(windowLength.toMinutes())
                    .withSecond(0)
                    .withNano(0);
    }

    private String convert2Field(LocalDateTime time) {
        return time.format(formatter());
    }

    private LocalDateTime convert2Time(String field) {
        return LocalDateTime.parse(field, formatter());
    }

    private DateTimeFormatter formatter() {
        return DateTimeFormatter.ofPattern("yyyyMMddHHmm");
    }

    @Override
    public String getPrimary() {
        return primary;
    }

    @Override
    public Duration getWindowLength() {
        return windowLength;
    }

    @Override
    public Duration getSlidingInterval() {
        return slidingInterval;
    }

    @Override
    public String getCurrentWindowed() {
        return currentWindowed;
    }

    public static class Duration {

        private final int value;
        private final TimeUnit timeUnit;

        public Duration(int value, TimeUnit timeUnit) {
            this.value = value;
            this.timeUnit = timeUnit;
        }

        public long toMinutes() {
            return timeUnit.toMinutes(value);
        }

        public static Duration of(int milliseconds) {
            return new Duration(milliseconds, TimeUnit.MILLISECONDS);
        }

        public static Duration days(int days) {
            return new Duration(days, TimeUnit.DAYS);
        }

        public static Duration hours(int hours) {
            return new Duration(hours, TimeUnit.HOURS);
        }

        public static Duration minutes(int minutes) {
            return new Duration(minutes, TimeUnit.MINUTES);
        }

        @Override
        public String toString() {
            return "Duration{value=" + this.value + '}';
        }
    }
}
